# tabela de distribuição de frequência para variáveis contínuas -----------


#'Generate a frequency table to a continuous variable.
#'
#'Bin the continuous variable, and count the occurrence of each interval.
#'
#'@param data dataframe object
#'@param variable variable name as character
#'@param breaks either a integer number or vector of number. Defaults to `NULL`
#'@param labels label for each interval. Defaults to  interval names using
#'  brackes and parentheses. Defaults to `NULL`.
#'@param include_lowest logical value. If `TRUE`, the lower limit of interval
#'  belongs to the bin. Defaults to `TRUE`.
#'@param right logical value. If `TRUE`, the upper limit of interval belongs to
#'  the bin. Defaults to `FALSE`.
#'
#'@details if `breaks = NULL`, then `ceiling(1 + log2(n))`, where `n` is the
#'  sample size.
#'
#'
#'@returns a dataframe with distribution frequency
#'
#' @examples
#' tab_freq_cont(iris, "Sepal.Width")
#'@export
#'@import dplyr janitor rlang datasets
tab_freq_cont <- function(data, variable, breaks = NULL,
                          labels = NULL, include_lowest = TRUE,
                          right = FALSE) {


  if (is.null(breaks)) {
    bins <- ceiling(1 + log2(nrow(data)))
  } else if (is.numeric(breaks) && (length(breaks) == 1)) {
    bins <- ceiling(breaks)
  } else if (is.numeric(breaks) && (length(breaks) > 1)) {
    bins <- breaks
  } else {
    stop("Invalid value for the argument 'breaks'.")
  }

  vetor <- cut(
    dplyr::pull(data, {{ variable }}),
    breaks = bins,
    include_lowest = include_lowest,
    right = right
  )

  res <- data.frame(vetor) |>
    tabyl(vetor) |>
    adorn_totals() |>
    dplyr::rename({{ variable }} := vetor)

  if (!is.null(labels)) {
    res <- res |>
      dplyr::mutate({{ variable }} := c(labels, "Total"))
  }

  res
}
