% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cut_densityMaps_for_focal_time.R
\name{cut_densityMaps_for_focal_time}
\alias{cut_densityMaps_for_focal_time}
\title{Cut phylogenies and posterior probability mapping of each state for a given focal time in the past}
\usage{
cut_densityMaps_for_focal_time(densityMaps, focal_time, keep_tip_labels = TRUE)
}
\arguments{
\item{densityMaps}{List of objects of class \code{"densityMap"}, typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}}.
Each densityMap (see \code{\link[phytools:densityMap]{phytools::densityMap()}}) contains a phylogenetic tree and associated posterior probability mapping of a categorical trait.
The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{focal_time}{Numerical. The time, in terms of time distance from the present,
for which the tree and mapping must be cut. It must be smaller than the root age of the phylogeny.}

\item{keep_tip_labels}{Logical. Specify whether terminal branches with a single descendant tip must retained their initial \code{tip.label}. Default is \code{TRUE}.}
}
\value{
The function returns an updated list of objects as cut densityMaps of class \code{"densityMap"}.

Each densityMap object contains three elements:
\itemize{
\item \verb{$tree} element of classes \code{"simmap"} and \code{"phylo"}. This function updates and adds multiple useful sub-elements to the \verb{$tree} element.
\itemize{
\item \verb{$maps} An updated list of named numerical vectors. Provides the mapping of posterior probability of the state along each remaining edge.
\item \verb{$mapped.edge} An updated matrix. Provides the evolutionary time spent across posterior probabilities (columns) along the remaining edges (rows).
\item \verb{$root_age} Integer. Stores the age of the root of the tree.
\item \verb{$nodes_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_node_ID} to the \code{initial_node_ID}. Each row is a node.
\item \verb{$initial_nodes_ID} Vector of character strings. Provides the initial ID of internal nodes. Used to plot internal node IDs as labels with \code{\link[ape:nodelabels]{ape::nodelabels()}}.
\item \verb{$edges_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_edge_ID} to the \code{initial_edge_ID}. Each row is an edge/branch.
\item \verb{$initial_edges_ID} Vector of character strings. Provides the initial ID of edges/branches. Used to plot edge/branch IDs as labels with \code{\link[ape:nodelabels]{ape::edgelabels()}}.
}
\item \verb{$col} element describes the colors used to map each possible posterior probability value from 0 to 1000.
\item \verb{$states} element provide the name of the states. Here, the first value is the absence of the state labeled as "Not X" with X being the state.
The second value is the name of the state.
}

High posterior probability reflects high likelihood to harbor the state. Low probability reflects high likelihood to NOT harbor the state.
}
\description{
Cuts off all the branches of the phylogeny which are
younger than a specific time in the past (i.e. the \code{focal_time}).
Branches overlapping the \code{focal_time} are shorten to the \code{focal_time}.
Likewise, remove posterior probability mapping of the categorical trait for the cut off branches
by updating the \verb{$tree$maps} and \verb{$tree$mapped.edge} elements.
}
\details{
The phylogenetic tree is cut for a specific time in the past (i.e. the \code{focal_time}).

When a branch with a single descendant tip is cut and \code{keep_tip_labels = TRUE},
the leaf left is labeled with the tip.label of the unique descendant tip.

When a branch with a single descendant tip is cut and \code{keep_tip_labels = FALSE},
the leaf left is labeled with the node ID of the unique descendant tip.

In all cases, when a branch with multiple descendant tips (i.e., a clade) is cut,
the leaf left is labeled with the node ID of the MRCA of the cut-off clade.

The continuous trait mapping is updated accordingly by removing mapping associated with the cut off branches.
}
\examples{
# ----- Prepare data ----- #

# Load mammals phylogeny and data from the R package motmot, and implemented in deepSTRAPP
# Data source: Slater, 2013; DOI: 10.1111/2041-210X.12084
data("mammals", package = "deepSTRAPP")

# Obtain mammal tree
mammals_tree <- mammals$mammal.phy
# Convert mass data into categories
mammals_mass <- setNames(object = mammals$mammal.mass$mean,
                         nm = row.names(mammals$mammal.mass))[mammals_tree$tip.label]
mammals_data <- mammals_mass
mammals_data[seq_along(mammals_data)] <- "small"
mammals_data[mammals_mass > 5] <- "medium"
mammals_data[mammals_mass > 10] <- "large"
table(mammals_data)

\donttest{ # (May take several minutes to run)
# Produce densityMaps using stochastic character mapping based on an equal-rates (ER) Mk model
mammals_cat_data <- prepare_trait_data(tip_data = mammals_data, phylo = mammals_tree,
                                       trait_data_type = "categorical",
                                       evolutionary_models = "ER",
                                       nb_simulations = 100,
                                       plot_map = FALSE)

# Set focal time
focal_time <- 80

# Extract the density maps
mammals_densityMaps <- mammals_cat_data$densityMaps

# ----- Example 1: keep_tip_labels = TRUE ----- #

# Cut densityMaps to 80 Mya while keeping tip.label
# on terminal branches with a unique descending tip.
updated_mammals_densityMaps <- cut_densityMaps_for_focal_time(
    densityMaps = mammals_densityMaps,
    focal_time = focal_time,
    keep_tip_labels = TRUE)

## Plot density maps as overlay of all state posterior probabilities
# ?plot_densityMaps_overlay

# Plot initial density maps
plot_densityMaps_overlay(densityMaps = mammals_densityMaps, fsize = 0.5)
abline(v = max(phytools::nodeHeights(mammals_densityMaps[[1]]$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated/cut density maps
plot_densityMaps_overlay(densityMaps = updated_mammals_densityMaps, fsize = 0.8)

# ----- Example 2: keep_tip_labels = FALSE ----- #

# Cut densityMap to 80 Mya while NOT keeping tip.label
updated_mammals_densityMaps <- cut_densityMaps_for_focal_time(
    densityMaps = mammals_densityMaps,
    focal_time = focal_time,
    keep_tip_labels = FALSE)

# Plot initial density maps
plot_densityMaps_overlay(densityMaps = mammals_densityMaps, fsize = 0.5)
abline(v = max(phytools::nodeHeights(mammals_densityMaps[[1]]$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated/cut density maps
plot_densityMaps_overlay(densityMaps = updated_mammals_densityMaps, fsize = 0.8) }

}
\seealso{
\code{\link[=cut_phylo_for_focal_time]{cut_phylo_for_focal_time()}} \code{\link[=cut_densityMap_for_focal_time]{cut_densityMap_for_focal_time()}}
\code{\link[=extract_most_likely_trait_values_for_focal_time]{extract_most_likely_trait_values_for_focal_time()}}
\code{\link[=extract_most_likely_states_from_densityMaps_for_focal_time]{extract_most_likely_states_from_densityMaps_for_focal_time()}}
}
\author{
Maël Doré
}
