
<!-- README.md is generated from README.Rmd. Please edit that file -->

# Difference-in-Differences <img src="man/figures/logo.png" align="right" alt="" width="155" />

[![](http://cranlogs.r-pkg.org/badges/grand-total/did?color=blue)](https://cran.r-project.org/package=did)
[![](http://cranlogs.r-pkg.org/badges/last-month/did?color=blue)](https://cran.r-project.org/package=did)
[![](https://www.r-pkg.org/badges/version/did?color=blue)](https://cran.r-project.org/package=did)
[![](https://img.shields.io/badge/devel%20version-2.3.0-blue.svg)](https://github.com/bcallaway11/did)
[![CRAN
checks](https://badges.cranchecks.info/summary/did.svg)](https://cran.r-project.org/web/checks/check_results_did.html)
[![](https://img.shields.io/github/last-commit/bcallaway11/did.svg)](https://github.com/bcallaway11/did/commits/master)

<!-- README.md is generated from README.Rmd. Please edit that file -->

The **did** package contains tools for computing average treatment
effect parameters in a Difference-in-Differences setup allowing for

- More than two time periods

- Variation in treatment timing (i.e., units can become treated at
  different points in time)

- Treatment effect heterogeneity (i.e, the effect of participating in
  the treatment can vary across units and exhibit potentially complex
  dynamics, selection into treatment, or time effects)

- The parallel trends assumption holds only after conditioning on
  covariates

The main parameters are **group-time average treatment effects**. These
are the average treatment effect for a particular group (group is
defined by treatment timing) in a particular time period. These
parameters are a natural generalization of the average treatment effect
on the treated (ATT) which is identified in the textbook case with two
periods and two groups to the case with multiple periods.

Group-time average treatment effects are also natural building blocks
for more aggregated treatment effect parameters such as overall
treatment effects or event-study-type estimands.

## Getting Started

There has been some recent work on DiD with multiple time periods. The
**did** package implements the framework put forward in

- [Callaway, Brantly and Pedro H.C. Sant’Anna.
  "Difference-in-Differences with Multiple Time Periods." Journal of
  Econometrics, Vol. 225, No. 2, pp. 200-230,
  2021.](https://doi.org/10.1016/j.jeconom.2020.12.001) or
  [arXiv](https://arxiv.org/abs/1803.09015)

**Higher level discussions of issues are available in**

- [Our approach to DiD with multiple time
  periods](https://bcallaway11.github.io/did/articles/multi-period-did.html)

## Installation

You can install **did** from CRAN with:

``` r
install.packages("did")
```

or get the latest version from github with:

``` r
# install.packages("devtools")
devtools::install_github("bcallaway11/did")
```

## A short example

The following is a simplified example of the effect of states increasing
their minimum wages on county-level teen employment rates which comes
from [Callaway and Sant’Anna
(2021)](https://authors.elsevier.com/a/1cFzc15Dji4pnC).

- [More detailed examples are also
  available](https://bcallaway11.github.io/did/articles/did-basics.html)

A subset of the data is available in the package and can be loaded by

``` r
library(did)
data(mpdta)
```

The dataset contains 500 observations of county-level teen employment
rates from 2003-2007. Some states are first treated in 2004, some in
2006, and some in 2007 (see the paper for more details). The important
variables in the dataset are

- **lemp** This is the log of county-level teen employment. It is the
  outcome variable

- **first.treat** This is the period when a state first increases its
  minimum wage. It can be 2004, 2006, or 2007. It is the variable that
  defines *group* in this application

- **year** This is the year and is the *time* variable

- **countyreal** This is an id number for each county and provides the
  individual identifier in this panel data context

To estimate group-time average treatment effects, use the **att_gt**
function

``` r
out <- att_gt(
  yname = "lemp",
  gname = "first.treat",
  idname = "countyreal",
  tname = "year",
  xformla = ~1,
  data = mpdta,
  est_method = "reg"
)
```

**att_gt** returns a class **MP** object. This has a lot of information,
but most importantly is has estimates of the group-time average
treatment effects and their standard errors. To see these, we can call
the **summary** function

``` r
summary(out)
#> 
#> Call:
#> att_gt(yname = "lemp", tname = "year", idname = "countyreal", 
#>     gname = "first.treat", xformla = ~1, data = mpdta, est_method = "reg")
#> 
#> Reference: Callaway, Brantly and Pedro H.C. Sant'Anna.  "Difference-in-Differences with Multiple Time Periods." Journal of Econometrics, Vol. 225, No. 2, pp. 200-230, 2021. <https://doi.org/10.1016/j.jeconom.2020.12.001>, <https://arxiv.org/abs/1803.09015> 
#> 
#> Group-Time Average Treatment Effects:
#>  Group Time ATT(g,t) Std. Error [95% Simult.  Conf. Band]  
#>   2004 2004  -0.0105     0.0237       -0.0727      0.0517  
#>   2004 2005  -0.0704     0.0311       -0.1518      0.0110  
#>   2004 2006  -0.1373     0.0393       -0.2401     -0.0344 *
#>   2004 2007  -0.1008     0.0355       -0.1938     -0.0078 *
#>   2006 2004   0.0065     0.0230       -0.0537      0.0668  
#>   2006 2005  -0.0028     0.0203       -0.0558      0.0503  
#>   2006 2006  -0.0046     0.0181       -0.0521      0.0429  
#>   2006 2007  -0.0412     0.0214       -0.0973      0.0148  
#>   2007 2004   0.0305     0.0164       -0.0124      0.0734  
#>   2007 2005  -0.0027     0.0170       -0.0474      0.0419  
#>   2007 2006  -0.0311     0.0188       -0.0804      0.0183  
#>   2007 2007  -0.0261     0.0168       -0.0701      0.0180  
#> ---
#> Signif. codes: `*' confidence band does not cover 0
#> 
#> P-value for pre-test of parallel trends assumption:  0.16812
#> Control Group:  Never Treated,  Anticipation Periods:  0
#> Estimation Method:  Outcome Regression
```

This provides estimates of group-time average treatment effects for all
groups in all time periods. Group-time average treatment effects are
identified when `t >= g` (these are post-treatment time periods for each
group), but **summary** reports them even in periods when `t < g` –
these can be used to pre-test for the parallel trends assumption. The
`P-value for pre-test of parallel trends assumption` is for a Wald
pre-test of the parallel trends assumption. Here the parallel trends
assumption would not be rejected at conventional significance levels.

It is often also convenient to plot the group-time average treatment
effects. This can be done using the **ggdid** command:

``` r
ggdid(out, ylim = c(-.25, .1))
```

<img src="man/figures/README-unnamed-chunk-9-1.png" width="90%" style="display: block; margin: auto;" />

The red dots in the plot are pre-treatment group-time average treatment
effects . Here they are provided with 95% simultaneous confidence
intervals. These are the estimates that can be interpreted as a pre-test
of the parallel trends assumption. The blue dots are post-treatment
group-time average treatment effects. Under the parallel trends
assumption, these can be interpreted as policy effects – here the effect
of the minimum wage on county-level teen employment due to increasing
the minimum wage.

**Event Studies**

Although in the current example it is pretty easy to directly interpret
the group-time average treatment effects, there are many cases where it
is convenient to aggregate the group-time average treatment effects into
a small number of parameters. A main type of aggregation is into an
*event study* plot.

To make an event study plot in the **did** package, one can use the
**aggte** function

``` r
es <- aggte(out, type = "dynamic")
```

Just like for group-time average treatment effects, these can be
summarized and plotted. First, the summary

``` r
summary(es)
#> 
#> Call:
#> aggte(MP = out, type = "dynamic")
#> 
#> Reference: Callaway, Brantly and Pedro H.C. Sant'Anna.  "Difference-in-Differences with Multiple Time Periods." Journal of Econometrics, Vol. 225, No. 2, pp. 200-230, 2021. <https://doi.org/10.1016/j.jeconom.2020.12.001>, <https://arxiv.org/abs/1803.09015> 
#> 
#> 
#> Overall summary of ATT's based on event-study/dynamic aggregation:  
#>      ATT    Std. Error     [ 95%  Conf. Int.]  
#>  -0.0772        0.0209    -0.1182     -0.0363 *
#> 
#> 
#> Dynamic Effects:
#>  Event time Estimate Std. Error [95% Simult.  Conf. Band]  
#>          -3   0.0305     0.0158       -0.0111      0.0721  
#>          -2  -0.0006     0.0128       -0.0342      0.0331  
#>          -1  -0.0245     0.0145       -0.0627      0.0137  
#>           0  -0.0199     0.0110       -0.0489      0.0090  
#>           1  -0.0510     0.0177       -0.0974     -0.0046 *
#>           2  -0.1373     0.0382       -0.2375     -0.0370 *
#>           3  -0.1008     0.0357       -0.1947     -0.0069 *
#> ---
#> Signif. codes: `*' confidence band does not cover 0
#> 
#> Control Group:  Never Treated,  Anticipation Periods:  0
#> Estimation Method:  Outcome Regression
```

The column `event time` is for each group relative to when they first
participate in the treatment. To give some examples, `event time=0`
corresponds to the *on impact* effect, and `event time=-1` is the
*effect* in the period before a unit becomes treated (checking that this
is equal to 0 is potentially useful as a pre-test).

To plot the event study, use **ggdid**

``` r
ggdid(es)
```

<img src="man/figures/README-unnamed-chunk-12-1.png" width="90%" style="display: block; margin: auto;" />

The figure here is very similar to the group-time average treatment
effects. Red dots are pre-treatment periods, blue dots are
post-treatment periods. The difference is that the x-axis is in event
time.

**Overall Effect of Participating in the Treatment**

The event study above reported an overall effect of participating in the
treatment. This was computed by averaging the average effects computed
at each length of exposure.

In many cases, a more general purpose overall treatment effect parameter
is give by computing the average treatment effect for each group, and
then averaging across groups. This sort of procedure provides an average
treatment effect parameter with a very similar interpretation to the
Average Treatment Effect on the Treated (ATT) in the two period and two
group case.

To compute this overall average treatment effect parameter, use

``` r
group_effects <- aggte(out, type = "group")
summary(group_effects)
#> 
#> Call:
#> aggte(MP = out, type = "group")
#> 
#> Reference: Callaway, Brantly and Pedro H.C. Sant'Anna.  "Difference-in-Differences with Multiple Time Periods." Journal of Econometrics, Vol. 225, No. 2, pp. 200-230, 2021. <https://doi.org/10.1016/j.jeconom.2020.12.001>, <https://arxiv.org/abs/1803.09015> 
#> 
#> 
#> Overall summary of ATT's based on group/cohort aggregation:  
#>     ATT    Std. Error     [ 95%  Conf. Int.]  
#>  -0.031        0.0129    -0.0563     -0.0057 *
#> 
#> 
#> Group Effects:
#>  Group Estimate Std. Error [95% Simult.  Conf. Band]  
#>   2004  -0.0797     0.0279       -0.1460     -0.0135 *
#>   2006  -0.0229     0.0165       -0.0621      0.0163  
#>   2007  -0.0261     0.0174       -0.0674      0.0153  
#> ---
#> Signif. codes: `*' confidence band does not cover 0
#> 
#> Control Group:  Never Treated,  Anticipation Periods:  0
#> Estimation Method:  Outcome Regression
```

Of particular interest is the `Overall ATT` in the results. Here, we
estimate that increasing the minimum wage decreased teen employment by
3.1% and the effect is marginally statistically significant.

## Additional Resources

We have provided several more vignettes that may be helpful for using
the **did** package

- [Getting Started with the did
  Package](https://bcallaway11.github.io/did/articles/did-basics.html)

- [Introduction to DiD with Multiple Time
  Periods](https://bcallaway11.github.io/did/articles/multi-period-did.html)

- [Pre-Testing in a DiD Setup using the did
  Package](https://bcallaway11.github.io/did/articles/pre-testing.html)

- [Writing Extensions to the did
  Package](https://bcallaway11.github.io/did/articles/extensions.html)
