% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/energy_ratios.R
\name{neb_func}
\alias{neb_func}
\title{Calculate Net Energy Burden (NEB)}
\usage{
neb_func(g, s, se = NULL, weights = NULL, aggregate = FALSE)
}
\arguments{
\item{g}{Numeric vector of gross income values}

\item{s}{Numeric vector of energy spending values}

\item{se}{Optional numeric vector of effective energy spending (defaults to s)}

\item{weights}{Optional numeric vector of weights for aggregation (e.g., household counts).
When provided, uses Nh method: \code{1 / (1 + weighted.mean(nh, weights))}}

\item{aggregate}{Logical, if TRUE forces aggregation even without weights (uses unweighted mean).
Default FALSE for backwards compatibility.}
}
\value{
\itemize{
\item If \code{weights = NULL} and \code{aggregate = FALSE}: Numeric vector of individual NEB values (S/G)
\item If \code{weights} provided or \code{aggregate = TRUE}: Single aggregated NEB value via Nh method
}
}
\description{
Calculates Net Energy Burden with proper aggregation methodology via the
Net Energy Return (Nh) framework. For individual households, NEB = EB = S/G.
When aggregating across households (with weights), automatically uses the
Nh method to avoid 1-5\% aggregation errors.
}
\details{
\strong{Individual Level:} NEB = EB = S/G (mathematically identical)

\strong{Aggregation Modes:}
\enumerate{
\item \strong{No aggregation} (default): Returns vector of individual NEB values

\if{html}{\out{<div class="sourceCode">}}\preformatted{neb_func(income, spending)  # Returns vector
}\if{html}{\out{</div>}}
\item \strong{Weighted aggregation}: Automatically uses Nh method when weights provided

\if{html}{\out{<div class="sourceCode">}}\preformatted{neb_func(income, spending, weights = households)  # Returns single value
}\if{html}{\out{</div>}}
\item \strong{Unweighted aggregation}: Use \code{aggregate = TRUE} for simple mean

\if{html}{\out{<div class="sourceCode">}}\preformatted{neb_func(income, spending, aggregate = TRUE)  # Returns single value
}\if{html}{\out{</div>}}
}

\strong{Why Nh Method?} Avoids 1-5\% error from naive averaging:
\itemize{
\item \strong{CORRECT}: \code{neb_func(g, s, weights = w)} → Uses Nh internally
\item \strong{WRONG}: \code{weighted.mean(s/g, w)} → Introduces bias
}

The Nh method: \code{1 / (1 + weighted.mean(nh, weights))} where \code{nh = (g-s)/se}
uses arithmetic mean instead of harmonic mean, providing computational
simplicity and numerical stability.
}
\examples{
# Individual household - returns vector
neb_func(50000, 3000)  # 0.06
neb_func(c(30000, 50000), c(3000, 3500))  # c(0.10, 0.07)

# Aggregation with weights - returns single value (CORRECT method)
incomes <- c(30000, 50000, 75000)
spending <- c(3000, 3500, 4000)
households <- c(100, 150, 200)
neb_func(incomes, spending, weights = households)

# Unweighted aggregation
neb_func(incomes, spending, aggregate = TRUE)

# Comparison: naive mean (WRONG) vs Nh method (CORRECT)
neb_naive <- weighted.mean(spending/incomes, households)  # Biased
neb_correct <- neb_func(incomes, spending, weights = households)  # Correct
abs(neb_naive - neb_correct) / neb_correct  # ~1-5\% error
}
\seealso{
\code{\link[=ner_func]{ner_func()}} for the Net Energy Return (Nh) calculation

\code{\link[=energy_burden_func]{energy_burden_func()}} for simple EB without aggregation support
}
