#' CSEM: Lord Keats Method
#'
#' @description
#' Compute CSEM using the Lord Keats approach, which rescales Lord's
#' binomial-model CSEM using empirical KR-20 and KR-21 reliability estimates.
#'
#' @param x A data frame or matrix of item responses, with rows as persons
#'   and columns as items. Items are assumed to be dichotomous (0/1).
#'
#' @details
#' This function first computes Lord's CSEM under the binomial model via
#' \code{csem_lord(ni)}, where \code{ni = ncol(x)}. It then rescales the
#' resulting CSEM curve using the ratio
#' \deqn{
#'   \sqrt{\frac{1 - \text{KR-20}}{1 - \text{KR-21}}},
#' }
#' where KR-20 and KR-21 are computed from the observed data via
#' \code{kr20(x)} and \code{kr21(x)}, respectively.
#'
#' @return A list with:
#' \describe{
#'   \item{x}{Vector of raw scores from 0 to \code{ni}.}
#'   \item{csem}{Vector of CSEM values under the Lord Keats method.}
#' }
#'
#' @examples
#' data(data.u)
#' csem_lord_keats(data.u)
#'
#' @export
csem_lord_keats <- function(x) {

  # basic checks ---------------------------------------------------------------
  if (is.null(x)) stop("`x` must not be NULL.")
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  x <- stats::na.exclude(x)
  x <- as.matrix(x)

  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  ni <- ncol(x)
  np <- nrow(x)

  if (ni < 2L) stop("`x` must have at least 2 items.")
  if (np < 2L) stop("`x` must have at least 2 persons.")

  # Lord CSEM under binomial model --------------------------------------------
  lord_res <- csem_lord(ni)
  lord_csem <- lord_res$csem

  # reliability estimates ------------------------------------------------------
  r20 <- kr20(x)
  r21 <- kr21(x)

  if (r20 >= 1 || r20 <= 0) {
    stop("KR-20 must be between 0 and 1 for scaling; got: ", r20)
  }
  if (r21 >= 1 || r21 <= 0) {
    stop("KR-21 must be between 0 and 1 for scaling; got: ", r21)
  }
  if (isTRUE(all.equal(1 - r21, 0))) {
    stop("Denominator (1 - KR-21) is zero; Lord Keats scaling undefined.")
  }

  # Lord Keats scaling factor --------------------------------------------------
  scale_factor <- sqrt((1 - r20) / (1 - r21))

  # apply scaling to Lord CSEM -------------------------------------------------
  csemx <- abs(lord_csem) * scale_factor
  # abs() just to be safe numerically; theoretically lord_csem >= 0

  return(list(x = 0:ni, csem = csemx))
}


