% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothGeom.r
\name{smoothGeom,GVector-method}
\alias{smoothGeom,GVector-method}
\alias{smoothGeom}
\title{Smooth the geometry of a vector}
\usage{
\S4method{smoothGeom}{GVector}(x, method = "Hermite", dist = NULL, angle = 3)
}
\arguments{
\item{x}{A \code{GVector}.}

\item{method}{Character: Method used to smooth line segments. Partial matching is used, and case does not matter:
\itemize{
\item \code{"Hermite"}: Hermite interpolation (default): Guarantees that the output vector always passes through the original points. This method adds points (possibly many) by constructing cubic splines with points approximately \code{dist} apart. The number of points can be reduced by specifying a smaller value of \code{angle}, which specifies the minimum angle between two successive line segments.
\item \code{"Chaiken"}: Chaiken's algorithm: Guarantees that the new vector always touches the midpoint of each original line segment. The points on the new line are at least \code{dist} apart.
}}

\item{dist}{Numeric > 0 of \code{NULL} (default): Minimum distance (see \code{method}). Units are in map units. If \code{NULL}, then 2\% of the minimum of the x-, y-, and z-extent will be used.}

\item{angle}{Numeric > 0: Maximum angle for the Hermite algorithm. Default is 3.}
}
\value{
A \code{GVector}.
}
\description{
\code{smoothGeom()} makes line segments of a vector appear less angular.
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

# Example data
madRivers <- fastData("madRivers")
rivers <- fast(madRivers)
soam <- rivers[rivers$NAM == "SOAMIANINA"] # select one river for illustration

### Simplify geometry (remove nodes)

vr <- simplifyGeom(soam, tolerance = 2000)
dp <- simplifyGeom(soam, tolerance = 2000, method = "dp")
dpr <- simplifyGeom(soam, tolerance = 2000, method = "dpr", prop = 0.5)
rw <- simplifyGeom(soam, tolerance = 2000, method = "rw")

plot(soam, col = "black", lwd = 3)
plot(vr, col = "blue", add = TRUE)
plot(dp, col = "red", add = TRUE)
plot(dpr, col = "chartreuse", add = TRUE)
plot(rw, col = "orange", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "Vertex reduction",
      "Douglas-Peucker",
      "Douglas-Peucker reduction",
      "Reumann-Witkam"
	),
	col = c("black", "blue", "red", "chartreuse", "orange"),
	lwd = c(3, 1, 1, 1, 1)
)

### Smooth geometry

hermite <- smoothGeom(soam, dist = 2000, angle = 3)
chaiken <- smoothGeom(soam, method = "Chaiken", dist = 2000)

plot(soam, col = "black", lwd = 2)
plot(hermite, col = "blue", add = TRUE)
plot(chaiken, col = "red", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "Hermite",
      "Chaiken"
	),
	col = c("black", "blue", "red"),
	lwd = c(2, 1, 1, 1, 1)
)

### Clean geometry

# Has no effect on this vector!
noDangs <- removeDangles(soam, tolerance = 10000)

plot(soam, col = "black", lwd = 2)
plot(noDangs, col = "red", add = TRUE)

legend("bottom",
   xpd = NA,
   legend = c(
	  "Original",
      "No dangles"
	),
	lwd = c(2, 1),
	col = c("black", "red")
)

}
}
\seealso{
\code{\link[=simplifyGeom]{simplifyGeom()}}, \code{\link[terra:simplify]{terra::simplifyGeom()}}, \link[=breakPolys]{geometry cleaning}, \strong{GRASS} manual page for tool \code{v.generalize} (see \code{grassHelp("v.generalize")})
}
