#' Compute Predictions from Fuzzy Linear Model
#'
#' @param object List. Result of fuzzy least squares regression containing `beta_hat`.
#' @param X_fuzzy List. Fuzzified predictor variables.
#' @return A list of fuzzy predictions.
#' @examples
#' # Example setup
#' X_fuzzy <- list(
#'   list(list(l = 1, x = 2, r = 3), list(l = 4, x = 5, r = 6)),
#'   list(list(l = 2, x = 3, r = 4), list(l = 5, x = 6, r = 7))
#' )
#' beta_hat <- c(0.5, 1.2, -0.8)  # Example regression coefficients
#' object <- list(beta_hat = beta_hat)
#'
#' # Compute predictions
#' predictions <- compute_pred(object, X_fuzzy)
#' print(head(predictions, 6))
#' @export
compute_pred <- function(object, X_fuzzy) {
  beta_hat <- object$beta_hat
  p <- length(beta_hat) - 1

  lapply(1:length(X_fuzzy), function(i) {
    pred <- list(l = beta_hat[1], x = beta_hat[1], r = beta_hat[1])
    for (j in 1:p) {
      pred <- fuzzy_add(pred, fuzzy_crisp_mult(beta_hat[j + 1], X_fuzzy[[i]][[j]]))
    }
    pred
  })
}


#' Compute Residuals for Fuzzy Linear Model
#'
#' @param Y_fuzzy List. Fuzzified observed response variables.
#' @param Y_pred List. Fuzzified predicted response variables.
#' @return A list of fuzzy residuals.
#' @examples
#' # Example setup
#' Y_fuzzy <- list(
#'   list(l = 2, x = 3, r = 4),
#'   list(l = 5, x = 6, r = 7)
#' )
#' Y_pred <- list(
#'   list(l = 1.5, x = 2.5, r = 3.5),
#'   list(l = 4.5, x = 5.5, r = 6.5)
#' )
#'
#' # Compute residuals
#' residuals <- compute_res(Y_fuzzy, Y_pred)
#' print(head(residuals, 6))
#' @export
compute_res <- function(Y_fuzzy, Y_pred) {
  lapply(seq_along(Y_fuzzy), function(i) {
    list(
      l = Y_fuzzy[[i]]$l - Y_pred[[i]]$l,
      x = Y_fuzzy[[i]]$x - Y_pred[[i]]$x,
      r = Y_fuzzy[[i]]$r - Y_pred[[i]]$r
    )
  })
}
