% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/futurize.R
\name{futurize}
\alias{futurize}
\alias{fz}
\title{Turn common R function calls into concurrent calls for parallel evaluation}
\usage{
futurize(
  expr,
  substitute = TRUE,
  options = futurize_options(...),
  ...,
  when = TRUE,
  eval = TRUE,
  envir = parent.frame()
)
}
\arguments{
\item{expr}{An \R expression, typically a function call to futurize.
If FALSE, then futurization is disabled, and if TRUE, it is
re-enabled.}

\item{substitute}{If TRUE, argument \code{expr} is
\code{\link[base]{substitute}()}:d, otherwise not.}

\item{options, \ldots}{Named options, passed to \code{\link[=futurize_options]{futurize_options()}},
controlling how futures are resolved.}

\item{when}{If TRUE (default), the expression is futurized, otherwise not.}

\item{eval}{If TRUE (default), the futurized expression is evaluated,
otherwise it is returned.}

\item{envir}{The \link{environment} from where global objects should be
identified.}
}
\value{
Returns the value of the evaluated expression \code{expr}.

If \code{expr} is TRUE or FALSE, then a logical is returned indicating
whether futurization was previously enabled or disabled.
}
\description{
\if{html}{\figure{futurize-logo.png}{options: style='float: right;' alt='logo' width='120'}}
}
\section{Expression unwrapping}{

The transpilation mechanism includes logic to "unwrap" expressions
enclosed in constructs such as \code{{ }}, \verb{( )}, \code{local()}, \code{I()},
\code{identity()}, \code{invisible()}, \code{suppressMessages()}, and
\code{suppressWarnings()}. The transpiler descends through wrapping
constructs until it finds a transpilable expression, avoiding the
need to place \code{futurize()} inside such constructs. This allows for
patterns like:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y <- \{
  lapply(xs, fcn)
\} |> suppressMessages() |> futurize()
}\if{html}{\out{</div>}}

avoiding having to write:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y <- \{
  lapply(xs, fcn) |> futurize()
\} |> suppressMessages()
}\if{html}{\out{</div>}}
}

\section{Conditional futurization}{

It is possible to control whether futurization should take place at
run-time. For example,

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y <- lapply(xs, fun) |> futurize(when = \{ length(xs) >= 10 \})
}\if{html}{\out{</div>}}

will be futurized, unless \code{length(xs)} is less than ten, in which case it is
evaluated as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y <- lapply(xs, fun)
}\if{html}{\out{</div>}}
}

\section{Disable and re-enable all futurization}{

It is possible to globally disable the effect of all \code{futurize()} calls
by calling \code{futurize(FALSE)}. The effect is as if \code{futurize()} was never
applied. For example,

\if{html}{\out{<div class="sourceCode r">}}\preformatted{futurize(FALSE)
y <- lapply(xs, fun) |> futurize()
}\if{html}{\out{</div>}}

evaluates as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y <- lapply(xs, fun)
}\if{html}{\out{</div>}}

To re-enable futurization, call \code{futurize(TRUE)}.
Please note that it is only the end-user that may control whether
futurization should be disabled and enabled. A package must \emph{never}
disable or enable futurization.
}

\examples{
xs <- list(1, 1:2, 1:2, 1:5)

# ------------------------------------------
# Base R apply functions
# ------------------------------------------
# Sequential lapply()
y <- lapply(X = xs, FUN = function(x) {
  sum(x)
})
   
# Parallel version
y <- lapply(X = xs, FUN = function(x) {
  sum(x)
}) |> futurize()
str(y)


# ------------------------------------------
# purrr map-reduce functions with pipes
# ------------------------------------------
if (require("purrr") && requireNamespace("furrr", quietly = TRUE)) {

# Sequential map()
y <- xs |> map(sum)
   
# Parallel version
y <- xs |> map(sum) |> futurize()
str(y)

} ## if (require ...)


# ------------------------------------------
# foreach map-reduce functions
# ------------------------------------------
if (require("foreach") && requireNamespace("doFuture", quietly = TRUE)) {

# Sequential foreach()
y <- foreach(x = xs) \%do\% {
  sum(x)
}
   
# Parallel version
y <- foreach(x = xs) \%do\% {
  sum(x)
} |> futurize()
str(y)


# Sequential times()
y <- times(3) \%do\% rnorm(1)
str(y)
   
# Parallel version
y <- times(3) \%do\% rnorm(1) |> futurize()
str(y)

} ## if (require ...)


# ------------------------------------------
# plyr map-reduce functions
# ------------------------------------------
if (require("plyr") && requireNamespace("doFuture", quietly = TRUE)) {

# Sequential llply()
y <- llply(xs, sum)
   
# Parallel version
y <- llply(xs, sum) |> futurize()
str(y)

} ## if (require ...)
}
