% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pathway_volcano.R
\name{pathway_volcano}
\alias{pathway_volcano}
\title{Volcano Plot for Pathway Differential Abundance Analysis}
\usage{
pathway_volcano(
  daa_results,
  fc_col = "log2_fold_change",
  p_col = "p_adjust",
  label_col = "pathway_name",
  fc_threshold = 1,
  p_threshold = 0.05,
  label_top_n = 10,
  point_size = 2,
  point_alpha = 0.6,
  colors = c(Down = "#3182bd", `Not Significant` = "grey60", Up = "#de2d26"),
  show_threshold_lines = TRUE,
  title = "Volcano Plot: Pathway Differential Abundance",
  x_lab = "log2 Fold Change",
  y_lab = "-log10(Adjusted P-value)"
)
}
\arguments{
\item{daa_results}{A data frame containing differential abundance analysis results,
typically from \code{\link{pathway_daa}}. Must contain columns for fold change,
p-values, and optionally pathway names.}

\item{fc_col}{Character string specifying the column name for log2 fold change values.
Default is "log2_fold_change". Legacy name "log2FoldChange" is also accepted.}

\item{p_col}{Character string specifying the column name for adjusted p-values.
Default is "p_adjust".}

\item{label_col}{Character string specifying the column name for pathway labels.
Default is "pathway_name". If NULL, no labels will be shown.}

\item{fc_threshold}{Numeric. Absolute fold change threshold for significance.
Default is 1 (2-fold change). Pathways with |log2FC| > fc_threshold are considered
biologically significant.}

\item{p_threshold}{Numeric. P-value threshold for statistical significance.
Default is 0.05.}

\item{label_top_n}{Integer. Number of top significant pathways to label.
Default is 10. Set to 0 to disable labels.}

\item{point_size}{Numeric. Size of points in the plot. Default is 2.}

\item{point_alpha}{Numeric. Transparency of points (0-1). Default is 0.6.}

\item{colors}{Named character vector with colors for "Down", "Not Significant", and "Up".
Default uses blue for down-regulated, grey for non-significant, and red for up-regulated.}

\item{show_threshold_lines}{Logical. Whether to show dashed lines for fold change
and p-value thresholds. Default is TRUE.}

\item{title}{Character string for plot title. Default is
"Volcano Plot: Pathway Differential Abundance".}

\item{x_lab}{Character string for x-axis label. Default is "log2 Fold Change".}

\item{y_lab}{Character string for y-axis label. Default is "-log10(Adjusted P-value)".}
}
\value{
A ggplot2 object that can be further customized or saved.
}
\description{
Creates a volcano plot to visualize the results of differential abundance analysis,
showing both statistical significance (-log10 p-value) and effect size (log2 fold change).
}
\details{
The volcano plot is a scatter plot that shows statistical significance (y-axis)
versus fold change (x-axis). Points are colored by significance:
\itemize{
  \item \strong{Up (red)}: Pathways with p < p_threshold AND log2FC > fc_threshold
  \item \strong{Down (blue)}: Pathways with p < p_threshold AND log2FC < -fc_threshold
  \item \strong{Not Significant (grey)}: All other pathways
}

The function automatically labels the top N most significant pathways using
\code{ggrepel::geom_text_repel()} if the ggrepel package is installed.
}
\examples{
\dontrun{
library(ggpicrust2)
library(tibble)

# Load example data
data("ko_abundance")
data("metadata")

# Convert KO to KEGG abundance
kegg_abundance <- ko2kegg_abundance(data = ko_abundance)

# Run differential abundance analysis
daa_results <- pathway_daa(
  abundance = kegg_abundance,
  metadata = metadata,
  group = "Environment",
  daa_method = "LinDA"
)

# Annotate results
daa_annotated <- pathway_annotation(
  pathway = "KO",
  ko_to_kegg = TRUE,
  daa_results_df = daa_results
)

# Create volcano plot
volcano_plot <- pathway_volcano(daa_annotated)
print(volcano_plot)

# Customize the plot
volcano_plot <- pathway_volcano(
  daa_annotated,
  fc_threshold = 0.5,
  p_threshold = 0.01,
  label_top_n = 15,
  colors = c("Down" = "darkblue", "Not Significant" = "lightgrey", "Up" = "darkred")
)
}

}
\seealso{
\code{\link{pathway_daa}}, \code{\link{pathway_annotation}},
  \code{\link{pathway_errorbar}}
}
