% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prune.R
\name{rm_data}
\alias{rm_data}
\alias{rm_data.git_repository}
\title{Remove Data Files From Git2rdata Objects}
\usage{
rm_data(root = ".", path = NULL, recursive = TRUE, ...)

\method{rm_data}{git_repository}(
  root,
  path = NULL,
  recursive = TRUE,
  ...,
  stage = FALSE,
  type = c("unmodified", "modified", "ignored", "all")
)
}
\arguments{
\item{root}{The root of a project. Can be a file path or a \code{git-repository}.
Defaults to the current working directory (\code{"."}).}

\item{path}{the directory in which to clean all the data files. The directory
is relative to \code{root}.}

\item{recursive}{remove files in subdirectories too.}

\item{...}{parameters used in some methods}

\item{stage}{stage the changes after removing the files. Defaults to FALSE.}

\item{type}{Defines the classes of files to remove. \code{unmodified} are files in
the git history and unchanged since the last commit. \code{modified} are files in
the git history and changed since the last commit. \code{ignored} refers to file
listed in a \code{.gitignore} file. Selecting \code{modified} will remove both
\code{unmodified} and \code{modified} data files. Selecting \code{ìgnored} will remove
\code{unmodified}, \code{modified} and \code{ignored} data files. \code{all} refers to all
visible data files, including \code{untracked} files.}
}
\value{
returns invisibly a vector of removed files names. The paths are
relative to \code{root}.
}
\description{
Remove the data (\code{.tsv}) file from all valid git2rdata objects at the \code{path}.
The metadata remains untouched. A warning lists any git2rdata object with
\strong{invalid} metadata. The function keeps any \code{.tsv} file with
invalid metadata or from non-git2rdata objects.

Use this function with caution since it will remove all valid data files
without asking for confirmation. We strongly recommend to use this
function on files under version control. See
\code{vignette("workflow", package = "git2rdata")} for some examples on how to use
this.
}
\examples{
## on file system

# create a directory
root <- tempfile("git2rdata-")
dir.create(root)

# store a dataframe as git2rdata object. Capture the result to minimise
# screen output
junk <- write_vc(
  iris[1:6, ], "iris", root, sorting = "Sepal.Length", digits = 6
)
# write a standard tab separate file (non git2rdata object)
write.table(iris, file = file.path(root, "standard.tsv"), sep = "\t")
# write a YAML file
yml <- list(
  authors = list(
   "Research Institute for Nature and Forest" = list(
       href = "https://www.inbo.be/en")))
yaml::write_yaml(yml, file = file.path(root, "_pkgdown.yml"))

# list the git2rdata objects
list_data(root)
# list the files
list.files(root, recursive = TRUE)

# remove all .tsv files from valid git2rdata objects
rm_data(root, path = ".")
# check the removal of the .tsv file
list.files(root, recursive = TRUE)
list_data(root)

# remove dangling git2rdata metadata files
prune_meta(root, path = ".")
# check the removal of the metadata
list.files(root, recursive = TRUE)
list_data(root)


## on git repo

# initialise a git repo using git2r
repo_path <- tempfile("git2rdata-repo-")
dir.create(repo_path)
repo <- git2r::init(repo_path)
git2r::config(repo, user.name = "Alice", user.email = "alice@example.org")

# store a dataframe
write_vc(
  iris[1:6, ], "iris", repo, sorting = "Sepal.Length", stage = TRUE,
  digits = 6
)
# check that the dataframe is stored
status(repo)
list_data(repo)

# commit the current version and check the git repo
commit(repo, "add iris data", session = TRUE)
status(repo)

# remove the data files from the repo
rm_data(repo, path = ".")
# check the removal
list_data(repo)
status(repo)

# remove dangling metadata
prune_meta(repo, path = ".")
# check the removal
list_data(repo)
status(repo)
}
\seealso{
Other storage: 
\code{\link{data_package}()},
\code{\link{display_metadata}()},
\code{\link{list_data}()},
\code{\link{prune_meta}()},
\code{\link{read_vc}()},
\code{\link{relabel}()},
\code{\link{rename_variable}()},
\code{\link{update_metadata}()},
\code{\link{verify_vc}()},
\code{\link{write_vc}()}
}
\concept{storage}
