% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation.R
\name{dglmstarma.sim}
\alias{dglmstarma.sim}
\title{Simulate spatial time-series based on double generalized linear models}
\usage{
dglmstarma.sim(
  ntime,
  parameters_mean,
  parameters_dispersion,
  model_mean,
  model_dispersion,
  mean_family = NULL,
  dispersion_link = c("log", "identity", "inverse"),
  wlist = NULL,
  mean_covariates = list(),
  dispersion_covariates = list(),
  pseudo_observations = c("deviance", "pearson"),
  wlist_past_mean = NULL,
  wlist_covariates = NULL,
  wlist_pseudo_obs = NULL,
  wlist_past_dispersion = NULL,
  wlist_covariates_dispersion = NULL,
  n_start = 100L,
  control = list()
)
}
\arguments{
\item{ntime}{Number of observation times to be simulated}

\item{parameters_mean}{a named list specifying the parameters of the model to be simulated:
\itemize{
\item \code{intercept} (numeric): Intercept parameter. If an inhomogeneous model is simulated, a value must be specified for each component of the time series.
\item \code{past_obs} (numeric matrix): Parameter values for the past observations.
\item \code{past_mean} (numeric matrix): Parameter values for the past means.
\item \code{covariates} (numeric matrix): Parameter values for the covariates.
}}

\item{parameters_dispersion}{a named list specifying the parameters of the dispersion model to be simulated, with the same possible elements as in \code{parameters_mean}.}

\item{model_mean}{a named list specifying the model for the linear predictor, which can be of the following elements:
\itemize{
\item \code{intercept} (character): 'homogenous' (default) for a homogenous model, i.e. the same intercept for all components, 'inhomogenous' for inhomogenous models, i.e. an individual intercept for each component.
\item \code{past_obs} (integer vector/binary matrix): Maximal spatial orders for the time lags in \code{past_obs_time_lags}. A binary matrix can be passed as an alternative, with the entry in row \eqn{i} and column \eqn{j} indicating whether the \eqn{(i - 1)}-spatial lag for the \eqn{j}-th time lag is included in the model. If not specified, no regression on past observations is performed.
\item \code{past_obs_time_lags} (optional integer vector) indicates the time lags for regression on past observations. Defaults are \code{seq(length(past_obs))} (for vectors) and \code{seq(ncol(past_obs))} (for a matrix)
\item \code{past_mean} (integer vector/binary matrix): Spatial orders for the regression on the (latent) feedback process. Values can be entered in the same format as in \code{past_obs}. If not specified, no regression to the feedback process is performed.
\item \code{past_mean_time_lags} (optional integer vector) indicates the time lags for regression on past values of the feedback process. Defaults are \code{seq(length(past_mean))} (for vectors) and \code{seq(ncol(past_mean))} (for a matrix)
\item \code{covariates} (integer vector/binary matrix) spatial orders for the covariate processes passed in the argument \code{covariates}. The values can be passed as in \code{past_obs} and \code{past_means}, where the \eqn{j}-th entry or column represents the \eqn{j}-th covariable. If no values are specified but covariates are included, the spatial order 0 is used by default, which corresponds to the first matrix in argument \code{wlist_covariates}.
}}

\item{model_dispersion}{a named list specifying the model for the dispersion linear predictor, with the same possible elements as in \code{model_mean}. Orders supplied in \code{past_obs} are applied to the pseudo-observations.}

\item{mean_family}{An object of class \code{stfamily} that specifies the marginal distributions of the observations and the link-function for the mean model.}

\item{dispersion_link}{Link function for the dispersion model. Possible values are \code{"log"} (default), \code{"identity"}, and \code{"inverse"}.}

\item{wlist}{A list of quadratic matrices, with the same dimension as the time series, which describe the spatial dependencies. Row-normalized matrices are recommended.}

\item{mean_covariates}{List of covariates included in the mean model, containing matrices or returns of the covariate functions of this package (see also \code{\link{TimeConstant}}, \code{\link{SpatialConstant}}).}

\item{dispersion_covariates}{List of covariates included in the dispersion model.}

\item{pseudo_observations}{Method to generate the  pseudo-observations for the dispersion model. Possible values are \code{"deviance"} (default) and \code{"pearson"}.}

\item{wlist_past_mean}{(Optional) List of matrices, which describes spatial dependencies for the past mean. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{wlist_covariates}{(Optional) List of matrices, which describes spatial dependencies for the covariates. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{wlist_pseudo_obs}{(Optional) List of matrices, which describes spatial dependencies for the pseudo-observations in the dispersion model. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{wlist_past_dispersion}{(Optional) List of matrices, which describes spatial dependencies for the past dispersion in the dispersion model. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{wlist_covariates_dispersion}{(Optional) List of matrices, which describes spatial dependencies for the covariates in the dispersion model. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{n_start}{Number of observations to be used for the burn-in period}

\item{control}{A list of parameters for controlling the fitting process. This list is passed to \code{\link{dglmstarma.control}}.}
}
\value{
a named list with the following elements:
\itemize{
\item \code{observations} (numeric matrix): The simulated time series
\item \code{link_values} (numeric matrix): The underlying linear predictor resulting from the model and simulation
\item \code{pseudo_observations} (numeric matrix): The pseudo-observations generated for the dispersion model
\item \code{dispersion_values} (numeric matrix): The dispersion values resulting from the dispersion model
\item \code{mean_model} (list): The mean model used for the simulation
\item \code{dispersion_model} (list): The dispersion model used for the simulation
\item \code{parameters_mean} (list): The true parameters used for the mean model
\item \code{parameters_dispersion} (list): The true parameters used for the dispersion model
}
}
\description{
Generates a simulated multivariate time series based on a GLM-like model (see \code{\link{dglmstarma}} for details)
}
\examples{
set.seed(42)
n_obs <- 200L
W <- generateW("rectangle", 100, 2, 10)
model_orders_mean <- list(intercept = "homogeneous", 
                          past_obs = 2, past_mean = 1, 
                          covariates = c(0, 0))
model_orders_dispersion <- list(intercept = "homogeneous", 
                                past_obs = 1, 
                                covariates = c(0, 0))

covariates_mean <- list(season = SpatialConstant(sin(2 * pi / 12 * seq(n_obs))),
                  location = TimeConstant(rnorm(100, sd = 0.81)))

covariates_dispersion <- list(season = SpatialConstant(sin(2 * pi / 24 * seq(n_obs))),
                        location = TimeConstant(runif(100)))

params_mean <- list(intercept = 0.6, 
                    past_mean = matrix(c(0.2, 0.1), nrow = 2), 
                    past_obs = matrix(c(0.2, 0.1, 0.05), nrow = 3), 
                    covariates = matrix(c(0.9, 0.2), ncol = 2))
params_dispersion <- list(intercept = 0.5, 
                    past_obs = matrix(c(0.5, 0.2), nrow = 2), 
                    covariates = matrix(c(0.1, 0.75), ncol = 2))
family <- vnormal(copula = "frank", copula_param = 2)
dglmstarma.sim(n_obs, params_mean, params_dispersion, model_orders_mean, 
              model_orders_dispersion, mean_family = family, 
              wlist = W, pseudo_observations = "deviance", 
              mean_covariates = covariates_mean, 
              dispersion_covariates = covariates_dispersion)
}
