% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/R6Model.R
\name{Model}
\alias{Model}
\title{A GLMM Model}
\description{
A GLMM Model

A GLMM Model
}
\details{
A generalised linear mixed model

See \link[glmmrBase]{glmmrBase-package} for a more in-depth guide.

The generalised linear mixed model is:

\deqn{Y \sim F(\mu,\sigma)}
\deqn{\mu = h^-1(X\beta + Zu)}
\deqn{u \sim MVN(0,D)}

where F is a distribution with scale parameter \deqn{\sigma}, h is a link function, X are the fixed effects with parameters \deqn{\beta}, Z is the random effect design matrix with multivariate Gaussian distributed effects u.
The class provides access to all of the elements of the model above and associated calculations and functions including model fitting, power analysis,
and various relevant matrices, including information matrices and related corrections. The object is an R6 class and so can serve as a parent class for extended functionality.

The currently supported families (links) are Gaussian (identity, log), Binomial (logit, log, probit, identity), Poisson (log, identity), Gamma (logit, identity, inverse), and Beta (logit).

This class provides model fitting functionality with a variety of stochastic maximum likelihood algorithms with and without restricted maximum likelihood corrections. A fast Laplace approximation is also included.
Small sample corrections are also provided including Kenward-Roger and Satterthwaite corrections.

Many calculations use the covariance matrix of the observations, such as the information matrix, which is used in power calculations and
other functions. For non-Gaussian models, the class uses the first-order approximation proposed by Breslow and Clayton (1993) based on the
marginal quasilikelihood:

\deqn{\Sigma = W^{-1} + ZDZ^T}

where \emph{W} is a diagonal matrix with the GLM iterated weights for each observation equal
to, for individual \emph{i} \eqn{\left( \frac{(\partial h^{-1}(\eta_i))}{\partial \eta_i}\right) ^2 Var(y|u)}
(see Table 2.1 in McCullagh and Nelder (1989)). The modification proposed by Zegers et al to the linear predictor to
improve the accuracy of approximations based on the marginal quasilikelihood is also available, see \code{use_attenuation()}.

See \href{https://github.com/samuel-watson/glmmrBase/blob/master/README.md}{glmmrBase} for a
detailed guide on model specification. A detailed vingette for this package is also available online\url{doi:10.48550/arXiv.2303.12657}.

\strong{Attenuation}
For calculations such as the information matrix, the first-order approximation to the covariance matrix
proposed by Breslow and Clayton (1993), described above, is used. The approximation is based on the
marginal quasilikelihood. Zegers, Liang, and Albert (1988) suggest that a better approximation to the
marginal mean is achieved by "attenuating" the linear predictor. Setting \code{use} equal to TRUE uses this
adjustment for calculations using the covariance matrix for non-linear models.

Calls the respective print methods of the linked covariance and mean function objects.

The matrices X and Z both have n rows, where n is the number of observations in the model/design.

Using \code{update_parameters()} is the preferred way of updating the parameters of the
mean or covariance objects as opposed to direct assignment, e.g. \code{self$covariance$parameters <- c(...)}.
The function calls check functions to automatically update linked matrices with the new parameters.

This function provides a large range of options for fitting GLMMs using stochastic algorithms. The function provides
fine control over all aspects of the algorithm, including method of sampling the random effects, convergence criteria,
optimisation methods, and so forth. For a fast and reliable alternative use the \code{fit()} function in this class, which is
recommended for most users.

\strong{Monte Carlo maximum likelihood}

Fits generalised linear mixed models using one of several algorithms: Markov Chain Newton
Raphson (MCNR), Markov Chain Expectation Maximisation (MCEM), or stochastic approximation expectation
maximisation (SAEM) with or without Polyak-Ruppert averaging. MCNR and MCEM are described by McCulloch (1997)
\url{doi:10.1080/01621459.1997.10473613}. For each iteration
of the algorithms the unobserved random effect terms (\eqn{\gamma}) are simulated
using Markov Chain Monte Carlo (MCMC) methods,
and then these values are conditioned on in the subsequent steps to estimate the covariance
parameters and the mean function parameters (\eqn{\beta}). SAEM uses a Robbins-Munroe approach to approximating
the likelihood and requires fewer MCMC samples and may have lower Monte Carlo error, see Jank (2006)\url{doi:10.1198/106186006X157469}.
The option \code{alpha} determines the rate at which succesive iterations "forget" the past and must be between 0.5 and 1. Higher values
will result in lower Monte Carlo error but slower convergence. The options \code{mcem.adapt} and \code{mcnr.adapt} will modify the number of MCMC samples during each step of model fitting
using the suggested values in Caffo, Jank, and Jones (2006)\url{doi:10.1111/j.1467-9868.2005.00499.x}
as the estimates converge. A Newton-Raphson step can be used for both fixed effect and covariance parameters with \code{mcnr2}.

The accuracy of the algorithm depends on the user specified tolerance. For higher levels of
tolerance, larger numbers of MCMC samples are likely need to sufficiently reduce Monte Carlo error. However,
the SAEM approach does overcome reduce the required samples. As such a lower number (20-50)
samples per iteration is normally sufficient to get convergence.

The default method uses MCMC sampling to generate samples of the random effects. As an alternative, samples can be
generated using a normal approximation with posterior mean and variance calculated directly by setting \code{mcmc.pkg = "analytic"}.

There are several stopping rules for the algorithm. Either the algorithm will terminate when succesive parameter estimates are
all within a specified tolerance of each other (\code{conv.criterion = 1}), or when there is a high probability that the estimated
log-likelihood has not been improved. This latter criterion can be applied to either the overall log-likelihood (\code{conv.criterion = 2}),
the likelihood just for the fixed effects (\code{conv.criterion = 3}), or both the likelihoods for the fixed effects and covariance parameters
(\code{conv.criterion = 4}; default).

The information printed to the console during model fitting can be controlled with the \code{self$set_trace()} function.

To provide weights for the model fitting, store them in self$weights. To set the number of
trials for binomial models, set self$trials.
}
\examples{

## ------------------------------------------------
## Method `Model$new`
## ------------------------------------------------

\dontshow{
setParallel(FALSE) 
}
# For more examples, see the examples for MCML.

#create a data frame describing a cross-sectional parallel cluster
#randomised trial
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
mod <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  data = df,
  family = stats::gaussian()
)

# We can also include the outcome data in the model initialisation. 
# For example, simulating data and creating a new object:
df$y <- mod$sim_data()

mod <- Model$new(
  formula = y ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  data = df,
  family = stats::gaussian()
)

# Here we will specify a cohort study
df <- nelder(~ind(20) * t(6))
df$int <- 0
df[df$t > 3, 'int'] <- 1

des <- Model$new(
  formula = ~ int + (1|gr(ind)),
  data = df,
  family = stats::poisson()
)
  
# or with parameter values specified
  
des <- Model$new(
  formula = ~ int + (1|gr(ind)),
  covariance = c(0.05),
  mean = c(1,0.5),
  data = df,
  family = stats::poisson()
)

#an example of a spatial grid with two time points

df <- nelder(~ (x(10)*y(10))*t(2))
spt_design <- Model$new(formula = ~ 1 + (1|ar0(t)*fexp(x,y)),
                        data = df,
                        family = stats::gaussian())

## ------------------------------------------------
## Method `Model$sim_data`
## ------------------------------------------------

df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
\dontshow{
setParallel(FALSE) # for the CRAN check
}
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)*ar0(t)),
  covariance = c(0.05,0.8),
  mean = c(rep(0,5),0.6),
  data = df,
  family = stats::binomial()
)
ysim <- des$sim_data()

## ------------------------------------------------
## Method `Model$update_parameters`
## ------------------------------------------------

\dontshow{
setParallel(FALSE) # for the CRAN check
}
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)*ar0(t)),
  data = df,
  family = stats::binomial()
)
des$update_parameters(cov.pars = c(0.1,0.9))

## ------------------------------------------------
## Method `Model$power`
## ------------------------------------------------

\dontshow{
setParallel(FALSE) # for the CRAN check
}
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  covariance = c(0.05,0.1),
  mean = c(rep(0,5),0.6),
  data = df,
  family = stats::gaussian(),
  var_par = 1
)
des$power() #power of 0.90 for the int parameter

## ------------------------------------------------
## Method `Model$MCML`
## ------------------------------------------------

\dontrun{
# Simulated trial data example
data(SimTrial,package = "glmmrBase")
model <- Model$new(
  formula = y ~ int + factor(t) - 1 + (1|gr(cl)*ar1(t)),
  data = SimTrial,
  family = gaussian()
)
glm3 <- model$MCML()

# Salamanders data example
data(Salamanders,package="glmmrBase")
model <- Model$new(
  mating~fpop:mpop-1+(1|gr(mnum))+(1|gr(fnum)),
  data = Salamanders,
  family = binomial()
)

# use MCEM + REML with 500 sampling iterations
glm2 <- model$MCML(method = "mcem", iter.sampling = 500, reml = TRUE)

# as an alternative, we will specify the variance parameters on the 
# log scale and use a fast fitting algorithm
# we will use two newton-raphson steps, and Normal approximation posteriors with 
# conjugate gradient descent
# the maximum number of iterations is increased as it takes 100-110 in this example
# we can also chain together the functions
# specify starting values for the covariance parameters to prevent potential fit failure
# with random initialisation on the log scale
glm3 <- Model$new(
  mating~fpop:mpop-1+(1|grlog(mnum))+(1|grlog(fnum)),
  data = Salamanders,
  family = binomial(),
  covariance = c(-0.5,-0.5)
)$MCML(method = "mcnr2", mcmc.pkg = "analytic", iter.sampling = 50, max.iter = 150)

# Example using simulated data
#create example data with six clusters, five time periods, and five people per cluster-period
df <- nelder(~(cl(6)*t(5)) > ind(5))
# parallel trial design intervention indicator
df$int <- 0
df[df$cl > 3, 'int'] <- 1 
# specify parameter values in the call for the data simulation below
des <- Model$new(
  formula= ~ factor(t) + int - 1 +(1|gr(cl)*ar0(t)),
  covariance = c(0.05,0.7),
  mean = c(rep(0,5),0.2),
  data = df,
  family = gaussian()
)
ysim <- des$sim_data() # simulate some data from the model
fit1 <- des$MCML(y = ysim) # Default model fitting with SAEM
# use MCNR instead 
fit2 <- des$MCML(y = ysim, method="mcnr")

# Non-linear model fitting example using the example provided by nlmer in lme4
data(Orange, package = "lme4")

# the lme4 example:
# startvec <- c(Asym = 200, xmid = 725, scal = 350)
# (nm1 <- lme4::nlmer(circumference ~ SSlogis(age, Asym, xmid, scal) ~ Asym|Tree,
#              Orange, start = startvec))

Orange <- as.data.frame(Orange)
Orange$Tree <- as.numeric(Orange$Tree)

# Here we can specify the model as a function. 

model <- Model$new(
  circumference ~ Asym/(1 + exp((xmid - (age))/scal)) - 1 + (Asym|gr(Tree)),
  data = Orange,
  family = gaussian(),
  mean = c(200,725,350),
  covariance = c(500),
  var_par = 50
)

# for this example, we will use MCEM with adaptive MCMC sample sizes

nfit <- model$MCML(method = "mcem.adapt", iter.sampling = 1000)

summary(nfit)


}

## ------------------------------------------------
## Method `Model$fit`
## ------------------------------------------------

# Simulated trial data example using REML
# specify covariance starting values to potential prevent fit failure with random intialisation
data(SimTrial,package = "glmmrBase")
fit1 <- Model$new(
  formula = y ~ int + factor(t) - 1 + (1|grlog(cl)*ar0log(t)),
  data = SimTrial,
  family = gaussian(),
  covariance = log(c(0.05,0.8)),
)$fit(reml = TRUE)

# Salamanders data example
data(Salamanders,package="glmmrBase")
# specify covariance starting values to prevent potential fit failure with random intialisation
model <- Model$new(
  mating~fpop:mpop-1+(1|grlog(mnum))+(1|grlog(fnum)),
  data = Salamanders,
  family = binomial(),
  covariance = c(-0.5, -0.5)
)

fit2 <- model$fit()

# Example using simulated data
#create example data with six clusters, five time periods, and five people per cluster-period
df <- nelder(~(cl(20)*t(10)) > ind(5))
# parallel trial design intervention indicator
df$int <- 0
df[df$cl > 10, 'int'] <- 1 
# specify parameter values in the call for the data simulation below
des <- Model$new(
  formula= ~ factor(t) + int - 1 +(1|grlog(cl)*ar0log(t)),
  covariance = log(c(0.15,0.7)),
  mean = c(rep(0,10),0.2),
  data = df,
  family = binomial()
)
ysim <- des$sim_data() # simulate some data from the model
des$update_y(ysim)
fit2 <- des$fit() 
}
\references{
Breslow, N. E., Clayton, D. G. (1993). Approximate Inference in Generalized Linear Mixed Models.
Journal of the American Statistical Association<, 88(421), 9–25. \url{doi:10.1080/01621459.1993.10594284}

McCullagh P, Nelder JA (1989). Generalized linear models, 2nd Edition. Routledge.

McCulloch CE (1997). “Maximum Likelihood Algorithms for Generalized Linear Mixed Models.”
Journal of the American statistical Association, 92(437), 162–170.\url{doi:10.2307/2291460}

Zeger, S. L., Liang, K.-Y., Albert, P. S. (1988). Models for Longitudinal Data: A Generalized Estimating Equation Approach.
Biometrics, 44(4), 1049.\url{doi:10.2307/2531734}
}
\seealso{
\link[glmmrBase]{nelder}, \link[glmmrBase]{MeanFunction}, \link[glmmrBase]{Covariance}

\link[glmmrBase]{Model}, \link[glmmrBase]{Covariance}, \link[glmmrBase]{MeanFunction}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{covariance}}{A \link[glmmrBase]{Covariance} object defining the random effects covariance.}

\item{\code{mean}}{A \link[glmmrBase]{MeanFunction} object, defining the mean function for the model, including the data and covariate design matrix X.}

\item{\code{family}}{One of the family function used in R's glm functions. See \link[stats]{family} for details}

\item{\code{weights}}{A vector indicting the weights for the observations.}

\item{\code{trials}}{For binomial family models, the number of trials for each observation. The default is 1 (bernoulli).}

\item{\code{formula}}{The formula for the model. May be empty if separate formulae are specified for the mean and covariance components.}

\item{\code{var_par}}{Scale parameter required for some distributions (Gaussian, Gamma, Beta).}

\item{\code{mcmc_options}}{There are three options for MCMC sampling that are specified in this list. Note that these
will be moved to arguments to the relevant function, and so are obsolete.
\itemize{
\item \code{warmup} The number of warmup iterations. Note that if using the internal HMC
sampler, this only applies to the first iteration of the MCML algorithm, as the
values from the previous iteration are carried over.
\item \code{samps} The number of MCMC samples drawn in the MCML algorithms. For
smaller tolerance values larger numbers of samples are required. For the internal
HMC sampler, larger numbers of samples are generally required than if using Stan since
the samples generally exhibit higher autocorrealtion, especially for more complex
covariance structures. For SAEM a small number is recommended as all samples are stored and used
from every iteration.
\code{chains} The number of MCMC chains for the Stan sampler.
}}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Model-use_attenuation}{\code{Model$use_attenuation()}}
\item \href{#method-Model-fitted}{\code{Model$fitted()}}
\item \href{#method-Model-residuals}{\code{Model$residuals()}}
\item \href{#method-Model-predict}{\code{Model$predict()}}
\item \href{#method-Model-new}{\code{Model$new()}}
\item \href{#method-Model-print}{\code{Model$print()}}
\item \href{#method-Model-n}{\code{Model$n()}}
\item \href{#method-Model-subset_rows}{\code{Model$subset_rows()}}
\item \href{#method-Model-sim_data}{\code{Model$sim_data()}}
\item \href{#method-Model-update_parameters}{\code{Model$update_parameters()}}
\item \href{#method-Model-information_matrix}{\code{Model$information_matrix()}}
\item \href{#method-Model-sandwich}{\code{Model$sandwich()}}
\item \href{#method-Model-small_sample_correction}{\code{Model$small_sample_correction()}}
\item \href{#method-Model-box}{\code{Model$box()}}
\item \href{#method-Model-power}{\code{Model$power()}}
\item \href{#method-Model-w_matrix}{\code{Model$w_matrix()}}
\item \href{#method-Model-dh_deta}{\code{Model$dh_deta()}}
\item \href{#method-Model-Sigma}{\code{Model$Sigma()}}
\item \href{#method-Model-MCML}{\code{Model$MCML()}}
\item \href{#method-Model-fit}{\code{Model$fit()}}
\item \href{#method-Model-sparse}{\code{Model$sparse()}}
\item \href{#method-Model-mcmc_sample}{\code{Model$mcmc_sample()}}
\item \href{#method-Model-importance_weights}{\code{Model$importance_weights()}}
\item \href{#method-Model-gradient}{\code{Model$gradient()}}
\item \href{#method-Model-partial_sigma}{\code{Model$partial_sigma()}}
\item \href{#method-Model-u}{\code{Model$u()}}
\item \href{#method-Model-log_likelihood}{\code{Model$log_likelihood()}}
\item \href{#method-Model-calculator_instructions}{\code{Model$calculator_instructions()}}
\item \href{#method-Model-marginal}{\code{Model$marginal()}}
\item \href{#method-Model-update_y}{\code{Model$update_y()}}
\item \href{#method-Model-set_trace}{\code{Model$set_trace()}}
\item \href{#method-Model-clone}{\code{Model$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-use_attenuation"></a>}}
\if{latex}{\out{\hypertarget{method-Model-use_attenuation}{}}}
\subsection{Method \code{use_attenuation()}}{
Sets the model to use or not use "attenuation" when calculating the first-order approximation to
the covariance matrix.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$use_attenuation(use)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{use}}{Logical indicating whether to use "attenuation".}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
None. Used for effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-fitted"></a>}}
\if{latex}{\out{\hypertarget{method-Model-fitted}{}}}
\subsection{Method \code{fitted()}}{
Return fitted values. Does not account for the random effects. For simulated values based
on resampling random effects, see also \code{sim_data()}. To predict the values including random effects at a new location see also
\code{predict()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$fitted(type = "link", X, u, sample = FALSE, sample_n = 100)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{One of either "\code{link}" for values on the scale of the link function, or "\code{response}"
for values on the scale of the response}

\item{\code{X}}{(Optional) Fixed effects matrix to generate fitted values}

\item{\code{u}}{(Optional) Random effects values at which to generate fitted values}

\item{\code{sample}}{Logical. If TRUE then the parameters will be re-sampled from their sampling distribution. Currently only works
with existing X matrix and not user supplied matrix X and this will also ignore any provided random effects.}

\item{\code{sample_n}}{Integer. If sample is TRUE, then this is the number of samples.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Fitted values as either a vector or matrix depending on the number of samples
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-residuals"></a>}}
\if{latex}{\out{\hypertarget{method-Model-residuals}{}}}
\subsection{Method \code{residuals()}}{
Generates the residuals for the model

Generates one of several types of residual for the model. If conditional = TRUE then
the residuals include the random effects, otherwise only the fixed effects are included. For type,
there are raw, pearson, and standardized residuals. For conditional residuals a matrix is returned
with each column corresponding to a sample of the random effects.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$residuals(type = "standardized", conditional = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{Either "standardized", "raw" or "pearson"}

\item{\code{conditional}}{Logical indicating whether to condition on the random effects (TRUE) or not (FALSE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix with either one column is conditional is false, or with number of columns corresponding
to the number of MCMC samples.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-predict"></a>}}
\if{latex}{\out{\hypertarget{method-Model-predict}{}}}
\subsection{Method \code{predict()}}{
Generate predictions at new values

Generates predicted values using a new data set to specify covariance
values and values for the variables that define the covariance function.
The function will return a list with the linear predictor, conditional
distribution of the new random effects term conditional on the current estimates
of the random effects, and some simulated values of the random effects if requested.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$predict(newdata, offset = rep(0, nrow(newdata)), m = 0)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{newdata}}{A data frame specifying the new data at which to generate predictions}

\item{\code{offset}}{Optional vector of offset values for the new data}

\item{\code{m}}{Number of samples of the random effects to draw}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list with the linear predictor, parameters (mean and covariance matrices) for
the conditional distribution of the random effects, and any random effect samples.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-new"></a>}}
\if{latex}{\out{\hypertarget{method-Model-new}{}}}
\subsection{Method \code{new()}}{
Create a new Model object. Typically, a model is generated from a formula and data. However, it can also be
generated from a previous model fit.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$new(
  formula,
  covariance,
  mean,
  data = NULL,
  family = NULL,
  var_par = NULL,
  offset = NULL,
  weights = NULL,
  trials = NULL,
  model_fit = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{formula}}{A model formula containing fixed and random effect terms. The formula can be one way (e.g. \code{~ x + (1|gr(cl))}) or
two-way (e.g. \code{y ~ x + (1|gr(cl))}). One way formulae will generate a valid model enabling data simulation, matrix calculation,
and power, etc. Outcome data can be passed directly to model fitting functions, or updated later using member function \code{update_y()}.
For binomial models, either the syntax \code{cbind(y, n-y)} can be used for outcomes, or just \code{y} and the number of trials passed to the argument
\code{trials} described below.}

\item{\code{covariance}}{(Optional) Either a \link[glmmrBase]{Covariance} object, an equivalent list of arguments
that can be passed to \code{Covariance} to create a new object, or a vector of parameter values. At a minimum the list must specify a formula.
If parameters are not included then they are initialised to 0.5.}

\item{\code{mean}}{(Optional) Either a \link[glmmrBase]{MeanFunction} object, an equivalent list of arguments
that can be passed to \code{MeanFunction} to create a new object, or a vector of parameter values. At a minimum the list must specify a formula.
If parameters are not included then they are initialised to 0.}

\item{\code{data}}{A data frame with the data required for the mean function and covariance objects. This argument
can be ignored if data are provided to the covariance or mean arguments either via \code{Covariance} and \code{MeanFunction}
object, or as a member of the list of arguments to both \code{covariance} and \code{mean}.}

\item{\code{family}}{A family object expressing the distribution and link function of the model, see \link[stats]{family}. Currently accepts \link[stats]{binomial},
\link[stats]{gaussian}, \link[stats]{Gamma}, \link[stats]{poisson}, \link[glmmrBase]{Beta}, and \link[glmmrBase]{Quantile}.}

\item{\code{var_par}}{(Optional) Scale parameter required for some distributions, including Gaussian. Default is NULL.}

\item{\code{offset}}{(Optional) A vector of offset values. Optional - could be provided to the argument to mean instead.}

\item{\code{weights}}{(Optional) A vector of weights.}

\item{\code{trials}}{(Optional) For binomial family models, the number of trials for each observation. If it is not set, then it will
default to 1 (a bernoulli model).}

\item{\code{model_fit}}{(optional) A \code{mcml} model fit resulting from a call to \code{MCML} or \code{LA}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new Model class object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontshow{
setParallel(FALSE) 
}
# For more examples, see the examples for MCML.

#create a data frame describing a cross-sectional parallel cluster
#randomised trial
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
mod <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  data = df,
  family = stats::gaussian()
)

# We can also include the outcome data in the model initialisation. 
# For example, simulating data and creating a new object:
df$y <- mod$sim_data()

mod <- Model$new(
  formula = y ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  data = df,
  family = stats::gaussian()
)

# Here we will specify a cohort study
df <- nelder(~ind(20) * t(6))
df$int <- 0
df[df$t > 3, 'int'] <- 1

des <- Model$new(
  formula = ~ int + (1|gr(ind)),
  data = df,
  family = stats::poisson()
)
  
# or with parameter values specified
  
des <- Model$new(
  formula = ~ int + (1|gr(ind)),
  covariance = c(0.05),
  mean = c(1,0.5),
  data = df,
  family = stats::poisson()
)

#an example of a spatial grid with two time points

df <- nelder(~ (x(10)*y(10))*t(2))
spt_design <- Model$new(formula = ~ 1 + (1|ar0(t)*fexp(x,y)),
                        data = df,
                        family = stats::gaussian())
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-print"></a>}}
\if{latex}{\out{\hypertarget{method-Model-print}{}}}
\subsection{Method \code{print()}}{
Print method for \code{Model} class
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$print()}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-n"></a>}}
\if{latex}{\out{\hypertarget{method-Model-n}{}}}
\subsection{Method \code{n()}}{
Returns the number of observations in the model
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$n(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-subset_rows"></a>}}
\if{latex}{\out{\hypertarget{method-Model-subset_rows}{}}}
\subsection{Method \code{subset_rows()}}{
Subsets the design keeping specified observations only

Given a vector of row indices, the corresponding rows will be kept and the
other rows will be removed from the mean function and covariance
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$subset_rows(index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{Integer or vector integers listing the rows to keep}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The function updates the object and nothing is returned.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-sim_data"></a>}}
\if{latex}{\out{\hypertarget{method-Model-sim_data}{}}}
\subsection{Method \code{sim_data()}}{
Generates a realisation of the design

Generates a single vector of outcome data based upon the
specified GLMM design.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$sim_data(type = "y")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{Either 'y' to return just the outcome data, 'data'
to return a data frame with the simulated outcome data alongside the model data,
or 'all', which will return a list with simulated outcomes y, matrices X and Z,
parameters beta, and the values of the simulated random effects.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Either a vector, a data frame, or a list
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
\dontshow{
setParallel(FALSE) # for the CRAN check
}
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)*ar0(t)),
  covariance = c(0.05,0.8),
  mean = c(rep(0,5),0.6),
  data = df,
  family = stats::binomial()
)
ysim <- des$sim_data()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-update_parameters"></a>}}
\if{latex}{\out{\hypertarget{method-Model-update_parameters}{}}}
\subsection{Method \code{update_parameters()}}{
Updates the parameters of the mean function and/or the covariance function
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$update_parameters(mean.pars = NULL, cov.pars = NULL, var.par = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mean.pars}}{(Optional) Vector of new mean function parameters}

\item{\code{cov.pars}}{(Optional) Vector of new covariance function(s) parameters}

\item{\code{var.par}}{(Optional) A scalar value for var_par}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontshow{
setParallel(FALSE) # for the CRAN check
}
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)*ar0(t)),
  data = df,
  family = stats::binomial()
)
des$update_parameters(cov.pars = c(0.1,0.9))
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-information_matrix"></a>}}
\if{latex}{\out{\hypertarget{method-Model-information_matrix}{}}}
\subsection{Method \code{information_matrix()}}{
Generates the information matrix of the mixed model GLS estimator (X'S^-1X). The inverse of this matrix is an
estimator for the variance-covariance matrix of the fixed effect parameters. For various small sample corrections
see \code{small_sample_correction()} and \code{box()}. For models with non-linear functions of fixed effect parameters,
a correction to the Hessian matrix is required, which is automatically calculated or optionally returned or disabled.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$information_matrix(include.re = FALSE, theta = FALSE, oim = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{include.re}}{logical indicating whether to return the information matrix including the random effects components (TRUE),
or the mixed model information matrix for beta only (FALSE).}

\item{\code{theta}}{Logical. If TRUE the function will return the variance-coviariance matrix for the covariance parameters and ignore the first argument. Otherwise, the fixed effect
parameter information matrix is returned.}

\item{\code{oim}}{Logical. If TRUE, returns the observed information matrix for both beta and theta, disregarding other arguments to the function.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-sandwich"></a>}}
\if{latex}{\out{\hypertarget{method-Model-sandwich}{}}}
\subsection{Method \code{sandwich()}}{
Returns the robust sandwich variance-covariance matrix for the fixed effect parameters
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$sandwich()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A PxP matrix
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-small_sample_correction"></a>}}
\if{latex}{\out{\hypertarget{method-Model-small_sample_correction}{}}}
\subsection{Method \code{small_sample_correction()}}{
Returns a small sample correction. The option "KR" returns the Kenward-Roger bias-corrected variance-covariance matrix
for the fixed effect parameters and degrees of freedom. Option "KR2"  returns an improved correction given
in Kenward & Roger (2009) \url{doi:j.csda.2008.12.013}. Note, that the corrected/improved version is invariant
under reparameterisation of the covariance, and it will also make no difference if the covariance is linear
in parameters. Exchangeable covariance structures in this package (i.e. \code{gr()}) are parameterised in terms of
the variance rather than standard deviation, so the results will be unaffected. Option "sat" returns the "Satterthwaite"
correction, which only includes corrected degrees of freedom, along with the GLS standard errors.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$small_sample_correction(type, oim = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{Either "KR", "KR2", or "sat", see description.}

\item{\code{oim}}{Logical. If TRUE use the observed information matrix, otherwise use the expected information matrix}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A PxP matrix
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-box"></a>}}
\if{latex}{\out{\hypertarget{method-Model-box}{}}}
\subsection{Method \code{box()}}{
Returns the inferential statistics (F-stat, p-value) for a modified Box correction \url{doi:10.1002/sim.4072} for
Gaussian-identity models.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$box(y)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{y}}{Optional. If provided, will update the vector of outcome data. Otherwise it will use the data from
the previous model fit.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A data frame.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-power"></a>}}
\if{latex}{\out{\hypertarget{method-Model-power}{}}}
\subsection{Method \code{power()}}{
Estimates the power of the design described by the model using the square root
of the relevant element of the GLS variance matrix:

\deqn{(X^T\Sigma^{-1}X)^{-1}}

Note that this is equivalent to using the "design effect" for many
models.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$power(alpha = 0.05, two.sided = TRUE, alternative = "pos")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{alpha}}{Numeric between zero and one indicating the type I error rate.
Default of 0.05.}

\item{\code{two.sided}}{Logical indicating whether to use a two sided test}

\item{\code{alternative}}{For a one-sided test whether the alternative hypothesis is that the
parameter is positive "pos" or negative "neg"}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A data frame describing the parameters, their values, expected standard
errors and estimated power.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontshow{
setParallel(FALSE) # for the CRAN check
}
df <- nelder(~(cl(10)*t(5)) > ind(10))
df$int <- 0
df[df$cl > 5, 'int'] <- 1
des <- Model$new(
  formula = ~ factor(t) + int - 1 + (1|gr(cl)) + (1|gr(cl,t)),
  covariance = c(0.05,0.1),
  mean = c(rep(0,5),0.6),
  data = df,
  family = stats::gaussian(),
  var_par = 1
)
des$power() #power of 0.90 for the int parameter
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-w_matrix"></a>}}
\if{latex}{\out{\hypertarget{method-Model-w_matrix}{}}}
\subsection{Method \code{w_matrix()}}{
Returns the diagonal of the matrix W used to calculate the covariance matrix approximation
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$w_matrix()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A vector with values of the glm iterated weights
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-dh_deta"></a>}}
\if{latex}{\out{\hypertarget{method-Model-dh_deta}{}}}
\subsection{Method \code{dh_deta()}}{
Returns the derivative of the link function with respect to the linear preditor
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$dh_deta()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A vector
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-Sigma"></a>}}
\if{latex}{\out{\hypertarget{method-Model-Sigma}{}}}
\subsection{Method \code{Sigma()}}{
Returns the (approximate) covariance matrix of y

Returns the covariance matrix Sigma. For non-linear models this is an approximation. See Details.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$Sigma(inverse = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{inverse}}{Logical indicating whether to provide the covariance matrix or its inverse}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-MCML"></a>}}
\if{latex}{\out{\hypertarget{method-Model-MCML}{}}}
\subsection{Method \code{MCML()}}{
Stochastic Maximum Likelihood model fitting
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$MCML(
  y = NULL,
  method = "saem",
  tol = 0.01,
  max.iter = 50,
  se = "gls",
  oim = FALSE,
  reml = FALSE,
  mcmc.pkg = "rstan",
  se.theta = TRUE,
  iter.warmup = NULL,
  iter.sampling = NULL,
  chains = NULL,
  lower.bound = NULL,
  upper.bound = NULL,
  lower.bound.theta = NULL,
  upper.bound.theta = NULL,
  alpha = 0.8,
  convergence.prob = 0.95,
  pr.average = FALSE,
  bf.tol = 10,
  bf.hist = 10,
  bf.k0 = 10,
  importance = FALSE,
  conv.criterion = 2,
  skip.theta = FALSE,
  constr.zero = 1
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{y}}{Optional. A numeric vector of outcome data. If this is not provided then either the outcome must have been specified when
initialising the Model object, or the outcome data has been updated using member function \code{update_y()}}

\item{\code{method}}{The MCML algorithm to use, either \code{mcem} or \code{mcnr}, \code{mcnr2}, or \code{saem} see Details. Default is \code{saem}. \code{mcem.adapt}, \code{mcnr2.adapt} and \code{mcnr.adapt} will use adaptive
MCMC sample sizes starting small and increasing to the the maximum value specified in \code{mcmc_options$sampling}, which may result in faster convergence. \code{saem} uses a
stochastic approximation expectation maximisation algorithm. MCMC samples are kept from all iterations and so a smaller number of samples are needed per iteration. The
qualifier \code{.dual} can also be added (e.g. \code{saem.dual}), which combines the fixed and covariance parameter estimation steps.}

\item{\code{tol}}{Numeric value, tolerance of the MCML algorithm, the maximum difference in parameter estimates
between iterations at which to stop the algorithm. If two values are provided then different tolerances will be
applied to the fixed effect and covariance parameters.}

\item{\code{max.iter}}{Integer. The maximum number of iterations of the MCML algorithm.}

\item{\code{se}}{String. Type of standard error and/or inferential statistics to return. Options are "gls" for GLS standard errors (the default),
"robust" for robust standard errors, "kr" for original Kenward-Roger bias corrected standard errors,
"kr2" for the improved Kenward-Roger correction, "sat" for Satterthwaite degrees of freedom correction (this is the same
degrees of freedom correction as Kenward-Roger, but with GLS standard errors), "box" to use a modified Box correction (does not return confidence intervals),
"bw" to use GLS standard errors with a between-within correction to the degrees of freedom, "bwrobust" to use robust
standard errors with between-within correction to the degrees of freedom.}

\item{\code{oim}}{Logical. If TRUE use the observed information matrix, otherwise use the expected information matrix for standard error and related calculations.}

\item{\code{reml}}{Logical. Whether to use a restricted maximum likelihood correction for fitting the covariance parameters}

\item{\code{mcmc.pkg}}{String. Either \code{rstan} to use rstan sampler, or
\code{analytic} to use a Normal approximation to the posterior with direct estimates of the posterior mean and variance. cmdstanr will compile the MCMC programs to the library folder the first time they are run,
so may not currently be an option for some users.}

\item{\code{se.theta}}{Logical. Whether to calculate the standard errors for the covariance parameters. This step is a slow part
of the calculation, so can be disabled if required in larger models. Has no effect for Kenward-Roger standard errors.}

\item{\code{iter.warmup}}{Integer. The number of warmup iterations for each MCMC run on each iteration of the algorithm. If this value is left as NULL then the value stored in self$mcmc_options$warmup will be used.}

\item{\code{iter.sampling}}{Integer. The number of sampling iterations for each MCMC run on each iteration of the algorithm. The default values have been selected to provide
relatively good convergence for the default SAEM algorithm, but may need to be increased for MCEM and MCNR. If an adaptive algorithm is used, then this is the maximum
number of iterations per MCMC run. If this value is left as NULL then the value stored in self$mcmc_options$samps will be used.}

\item{\code{chains}}{Integer. The number of MCMC chains to run in parallel. The default is one, which generally provides good results. If this value is left as NULL then the value stored in self$mcmc_options$chains will be used.}

\item{\code{lower.bound}}{Optional. Vector of lower bounds for the fixed effect parameters. To apply bounds use MCEM.}

\item{\code{upper.bound}}{Optional. Vector of upper bounds for the fixed effect parameters. To apply bounds use MCEM.}

\item{\code{lower.bound.theta}}{Optional. Vector of lower bounds for the covariance parameters (default is 0; negative values will cause an error)}

\item{\code{upper.bound.theta}}{Optional. Vector of upper bounds for the covariance parameters.}

\item{\code{alpha}}{If using SAEM then this parameter controls the step size. On each iteration i the step size is (1/alpha)^i, default is 0.8. Values around 0.5
will result in lower bias but slower convergence, values closer to 1 will result in higher convergence but potentially higher error.}

\item{\code{convergence.prob}}{Numeric value in (0,1) indicating the probability of convergence if using convergence criteria 2, 3, or 4.}

\item{\code{pr.average}}{Logical indicating whether to use Polyak-Ruppert averaging if using the SAEM algorithm (default is FALSE)}

\item{\code{bf.tol}}{Integer indicating the Bayes Factor convergence criterion}

\item{\code{bf.hist}}{Integer, the number of iterations in the running mean for the Bayes Factor convergence criterion}

\item{\code{bf.k0}}{Integer, the expected number of iterations to convergence of the Bayes Factor convergence criterion.}

\item{\code{importance}}{Logical. If TRUE and using the analytic samples, the u samples are weighted using an importance sampling step. If FALSE, it is equivalent to the Laplace
Approximation Gaussian distribution of the random effects.}

\item{\code{conv.criterion}}{Integer. The convergence criterion for the algorithm. 1 = the maximum difference between parameter estimates between iterations as defined by \code{tol},
2 = The probability of improvement in the overall log-likelihood is less than 1 - \code{convergence.prob}
3 = The probability of improvement in the log-likelihood for the fixed effects is less than 1 - \code{convergence.prob}
4 = The probabilities of improvement in the log-likelihood the fixed effects and covariance parameters are both less than 1 - \code{convergence.prob}}

\item{\code{skip.theta}}{Logical. If TRUE then the covariance parameter estimation step is skipped. This option is mainly used for testing, but may be useful
if covariance parameters are known.}

\item{\code{constr.zero}}{Scalar. A Soft sum-to-zero constraint can be forced on the random effects so that their sum is N(0,constr.zero*Q). Small values, e.g. 0.001
may be useful if there is possible identifiability issues for intercept terms, such as in more complex, or higher dimensional, random effects structures like spatial models.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{mcml} object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Simulated trial data example
data(SimTrial,package = "glmmrBase")
model <- Model$new(
  formula = y ~ int + factor(t) - 1 + (1|gr(cl)*ar1(t)),
  data = SimTrial,
  family = gaussian()
)
glm3 <- model$MCML()

# Salamanders data example
data(Salamanders,package="glmmrBase")
model <- Model$new(
  mating~fpop:mpop-1+(1|gr(mnum))+(1|gr(fnum)),
  data = Salamanders,
  family = binomial()
)

# use MCEM + REML with 500 sampling iterations
glm2 <- model$MCML(method = "mcem", iter.sampling = 500, reml = TRUE)

# as an alternative, we will specify the variance parameters on the 
# log scale and use a fast fitting algorithm
# we will use two newton-raphson steps, and Normal approximation posteriors with 
# conjugate gradient descent
# the maximum number of iterations is increased as it takes 100-110 in this example
# we can also chain together the functions
# specify starting values for the covariance parameters to prevent potential fit failure
# with random initialisation on the log scale
glm3 <- Model$new(
  mating~fpop:mpop-1+(1|grlog(mnum))+(1|grlog(fnum)),
  data = Salamanders,
  family = binomial(),
  covariance = c(-0.5,-0.5)
)$MCML(method = "mcnr2", mcmc.pkg = "analytic", iter.sampling = 50, max.iter = 150)

# Example using simulated data
#create example data with six clusters, five time periods, and five people per cluster-period
df <- nelder(~(cl(6)*t(5)) > ind(5))
# parallel trial design intervention indicator
df$int <- 0
df[df$cl > 3, 'int'] <- 1 
# specify parameter values in the call for the data simulation below
des <- Model$new(
  formula= ~ factor(t) + int - 1 +(1|gr(cl)*ar0(t)),
  covariance = c(0.05,0.7),
  mean = c(rep(0,5),0.2),
  data = df,
  family = gaussian()
)
ysim <- des$sim_data() # simulate some data from the model
fit1 <- des$MCML(y = ysim) # Default model fitting with SAEM
# use MCNR instead 
fit2 <- des$MCML(y = ysim, method="mcnr")

# Non-linear model fitting example using the example provided by nlmer in lme4
data(Orange, package = "lme4")

# the lme4 example:
# startvec <- c(Asym = 200, xmid = 725, scal = 350)
# (nm1 <- lme4::nlmer(circumference ~ SSlogis(age, Asym, xmid, scal) ~ Asym|Tree,
#              Orange, start = startvec))

Orange <- as.data.frame(Orange)
Orange$Tree <- as.numeric(Orange$Tree)

# Here we can specify the model as a function. 

model <- Model$new(
  circumference ~ Asym/(1 + exp((xmid - (age))/scal)) - 1 + (Asym|gr(Tree)),
  data = Orange,
  family = gaussian(),
  mean = c(200,725,350),
  covariance = c(500),
  var_par = 50
)

# for this example, we will use MCEM with adaptive MCMC sample sizes

nfit <- model$MCML(method = "mcem.adapt", iter.sampling = 1000)

summary(nfit)


}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-fit"></a>}}
\if{latex}{\out{\hypertarget{method-Model-fit}{}}}
\subsection{Method \code{fit()}}{
MCML model fitting with the fastest options

Uses double Newton-Raphson method (with or without REML for Gaussian models). For details on the algorithm see \code{MCML()}. This function
uses the fastest set of options, including specialised model fitting for linear models. Note that no random effect
samples are drawn for Gaussian models, but can be subsequently drawn using \code{mcmc_sample()}. It is recommended to use the log
version of the covariance functions with this method as the Newton-Raphson steps can lead to negative values otherwise.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$fit(
  niter = 100,
  max_iter = 30,
  tol = ifelse(self$family[[1]] == "gaussian" & self$family[[2]] == "identity", 1e-06,
    10),
  hist = 5,
  k0 = 10,
  reml = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{niter}}{Integer. Number of samples for the random effects, ignored for Gaussian models, see examples.}

\item{\code{max_iter}}{Integer. Maximum number of iterations.}

\item{\code{tol}}{Scalar. The tolerance for the convergence criterion. For GLMMs this is the tolerance for
the Bayes Factor convergence criterion, for Gaussian linear models the tolerance is the difference
in the log-likelihood between successive iterations.}

\item{\code{hist}}{Integer. The length of the running mean for the convergence criterion for non-Gaussian models.}

\item{\code{k0}}{Integer. The expected number of iterations until convergence.}

\item{\code{reml}}{Bool. For Gaussian models, whether to use REML or not.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{mcml} model fit object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# Simulated trial data example using REML
# specify covariance starting values to potential prevent fit failure with random intialisation
data(SimTrial,package = "glmmrBase")
fit1 <- Model$new(
  formula = y ~ int + factor(t) - 1 + (1|grlog(cl)*ar0log(t)),
  data = SimTrial,
  family = gaussian(),
  covariance = log(c(0.05,0.8)),
)$fit(reml = TRUE)

# Salamanders data example
data(Salamanders,package="glmmrBase")
# specify covariance starting values to prevent potential fit failure with random intialisation
model <- Model$new(
  mating~fpop:mpop-1+(1|grlog(mnum))+(1|grlog(fnum)),
  data = Salamanders,
  family = binomial(),
  covariance = c(-0.5, -0.5)
)

fit2 <- model$fit()

# Example using simulated data
#create example data with six clusters, five time periods, and five people per cluster-period
df <- nelder(~(cl(20)*t(10)) > ind(5))
# parallel trial design intervention indicator
df$int <- 0
df[df$cl > 10, 'int'] <- 1 
# specify parameter values in the call for the data simulation below
des <- Model$new(
  formula= ~ factor(t) + int - 1 +(1|grlog(cl)*ar0log(t)),
  covariance = log(c(0.15,0.7)),
  mean = c(rep(0,10),0.2),
  data = df,
  family = binomial()
)
ysim <- des$sim_data() # simulate some data from the model
des$update_y(ysim)
fit2 <- des$fit() 
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-sparse"></a>}}
\if{latex}{\out{\hypertarget{method-Model-sparse}{}}}
\subsection{Method \code{sparse()}}{
Set whether to use sparse matrix methods for model calculations and fitting.
By default the model does not use sparse matrix methods.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$sparse(sparse = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sparse}}{Logical indicating whether to use sparse matrix methods}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
None, called for effects
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-mcmc_sample"></a>}}
\if{latex}{\out{\hypertarget{method-Model-mcmc_sample}{}}}
\subsection{Method \code{mcmc_sample()}}{
Generate an MCMC sample of the random effects
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$mcmc_sample(mcmc.pkg = "rstan", scaled = TRUE, constr.zero = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mcmc.pkg}}{String. Either \code{analytic} for importance sampling from Gaussian posterior proposal, \code{cmdstan} for cmdstan
(requires the package \code{cmdstanr}), \code{rstan} to use rstan sampler (the default)}

\item{\code{scaled}}{Logical. The random effects are sampled from an N(0,I) distribution. If TRUE this function returns the random effects rescaled to N(0,D), otherwise it returns the original samples.}

\item{\code{constr.zero}}{Scalar. A Soft sum-to-zero constraint can be forced on the random effects so that their sum is N(0,constr.zero*Q). Small values, e.g. 0.001
may be useful if there is possible identifiability issues for intercept terms, such as in more complex, or higher dimensional, random effects structures like spatial models.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix of samples of the random effects
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-importance_weights"></a>}}
\if{latex}{\out{\hypertarget{method-Model-importance_weights}{}}}
\subsection{Method \code{importance_weights()}}{
Returns the importance weights for the random effect samples. If using MCMC then weights are all equal.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$importance_weights()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A vector of the weights
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-gradient"></a>}}
\if{latex}{\out{\hypertarget{method-Model-gradient}{}}}
\subsection{Method \code{gradient()}}{
The gradient of the log-likelihood with respect to either the random effects or
the model parameters. The random effects are on the N(0,I) scale, i.e. scaled by the
Cholesky decomposition of the matrix D. To obtain the random effects from the last
model fit, see member function \verb{$u}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$gradient(y, u, beta = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{y}}{(optional) Vector of outcome data, if not specified then data must have been set in another function.}

\item{\code{u}}{(optional) Vector of random effects scaled by the Cholesky decomposition of D}

\item{\code{beta}}{Logical. Whether the log gradient for the random effects (FALSE) or for the linear predictor parameters (TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A vector of the gradient
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-partial_sigma"></a>}}
\if{latex}{\out{\hypertarget{method-Model-partial_sigma}{}}}
\subsection{Method \code{partial_sigma()}}{
The partial derivatives of the covariance matrix Sigma with respect to the covariance
parameters. The function returns a list in order: Sigma, first order derivatives, second
order derivatives. The second order derivatives are ordered as the lower-triangular matrix
in column major order. Letting 'd(i)' mean the first-order partial derivative with respect
to parameter i, and d2(i,j) mean the second order derivative with respect to parameters i
and j, then if there were three covariance parameters the order of the output would be:
(sigma, d(1), d(2), d(3), d2(1,1), d2(1,2), d2(1,3), d2(2,2), d2(2,3), d2(3,3)).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$partial_sigma()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A list of matrices, see description for contents of the list.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-u"></a>}}
\if{latex}{\out{\hypertarget{method-Model-u}{}}}
\subsection{Method \code{u()}}{
Returns the sample of random effects from the last model fit, or updates the samples for the model.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$u(scaled = TRUE, u)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{scaled}}{Logical indicating whether the samples are on the N(0,I) scale (\code{scaled=FALSE}) or
N(0,D) scale (\code{scaled=TRUE})}

\item{\code{u}}{(optional) Matrix of random effect samples. If provided then the internal samples are replaced with these values. These samples should be N(0,I).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix of random effect samples
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-log_likelihood"></a>}}
\if{latex}{\out{\hypertarget{method-Model-log_likelihood}{}}}
\subsection{Method \code{log_likelihood()}}{
The log likelihood for the GLMM. The random effects can be left
unspecified. If no random effects are provided, and there was a previous model fit with the same data \code{y}
then the random effects will be taken from that model. If there was no
previous model fit then the random effects are assumed to be all zero.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$log_likelihood(y, u)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{y}}{A vector of outcome data}

\item{\code{u}}{An optional matrix of random effect samples. This can be a single column.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The log-likelihood of the model parameters
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-calculator_instructions"></a>}}
\if{latex}{\out{\hypertarget{method-Model-calculator_instructions}{}}}
\subsection{Method \code{calculator_instructions()}}{
Prints the internal instructions and data used to calculate the linear predictor.
Internally the class uses a reverse polish notation to store and
calculate different functions, including user-specified non-linear mean functions. This
function will print all the steps. Mainly used for debugging and determining how the
class has interpreted non-linear model specifications.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$calculator_instructions()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
None. Called for effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-marginal"></a>}}
\if{latex}{\out{\hypertarget{method-Model-marginal}{}}}
\subsection{Method \code{marginal()}}{
Calculates the marginal effect of variable x. There are several options for
marginal effect and several types of conditioning or averaging. The type of marginal
effect can be the derivative of the mean with respect to x (\code{dydx}), the expected
difference E(y|x=a)-E(y|x=b) (\code{diff}), or the expected log ratio log(E(y|x=a)/E(y|x=b)) (\code{ratio}).
Other fixed effect variables can be set at specific values (\code{at}), set at their mean values
(\code{atmeans}), or averaged over (\code{average}). Averaging over a fixed effects variable here means
using all observed values of the variable in the relevant calculation.
The random effects can similarly be set at their
estimated value (\code{re="estimated"}), set to zero (\code{re="zero"}), set to a specific value
(\code{re="at"}), or averaged over (\code{re="average"}). Estimates of the expected values over the random
effects are generated using MCMC samples. MCMC samples are generated either through
MCML model fitting or using \code{mcmc_sample}. In the absence of samples \code{average} and \code{estimated}
will produce the same result. The standard errors are calculated using the delta method with one
of several options for the variance matrix of the fixed effect parameters.
Several of the arguments require the names
of the variables as given to the model object. Most variables are as specified in the formula,
factor variables are specified as the name of the \code{variable_value}, e.g. \code{t_1}. To see the names
of the stored parameters and data variables see the member function \code{names()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$marginal(
  x,
  type,
  re,
  se,
  at = c(),
  atmeans = c(),
  average = c(),
  xvals = c(1, 0),
  atvals = c(),
  revals = c(),
  oim = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{String. Name of the variable to calculate the marginal effect for.}

\item{\code{type}}{String. Either \code{dydx} for derivative, \code{diff} for difference, or \code{ratio} for log ratio. See description.}

\item{\code{re}}{String. Either \code{estimated} to condition on estimated values, \code{zero} to set to zero, \code{at} to
provide specific values, or \code{average} to average over the random effects.}

\item{\code{se}}{String. Type of standard error to use, either \code{GLS} for the GLS standard errors, \code{KR} for
Kenward-Roger estimated standard errors, \code{KR2} for the improved Kenward-Roger correction (see \code{small_sample_correction()}),
or \code{robust} to use a robust sandwich estimator.}

\item{\code{at}}{Optional. A vector of strings naming the fixed effects for which a specified value is given.}

\item{\code{atmeans}}{Optional. A vector of strings naming the fixed effects that will be set at their mean value.}

\item{\code{average}}{Optional. A vector of strings naming the fixed effects which will be averaged over.}

\item{\code{xvals}}{Optional. A vector specifying the values of \code{a} and \code{b} for \code{diff} and \code{ratio}. The default is (1,0).}

\item{\code{atvals}}{Optional. A vector specifying the values of fixed effects specified in \code{at} (in the same order).}

\item{\code{revals}}{Optional. If \code{re="at"} then this argument provides a vector of values for the random effects.}

\item{\code{oim}}{Logical. If TRUE use the observed information matrix, otherwise use the expected information matrix for standard error and related calculations.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A named vector with elements \code{margin} specifying the point estimate and \code{se} giving the standard error.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-update_y"></a>}}
\if{latex}{\out{\hypertarget{method-Model-update_y}{}}}
\subsection{Method \code{update_y()}}{
Updates the outcome data y

Some functions require outcome data, which is by default set to all zero if no model fitting function
has been run. This function can update the interval y data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$update_y(y)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{y}}{Vector of outcome data}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
None. Called for effects
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-set_trace"></a>}}
\if{latex}{\out{\hypertarget{method-Model-set_trace}{}}}
\subsection{Method \code{set_trace()}}{
Controls the information printed to the console for other functions.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$set_trace(trace)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{trace}}{Integer, either 0 = no information, 1 = some information, 2 = all information}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
None. Called for effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Model-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Model-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Model$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
