% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/open.r
\name{h5_open}
\alias{h5_open}
\title{Create an HDF5 File Handle}
\usage{
h5_open(file)
}
\arguments{
\item{file}{Path to the HDF5 file. The file will be created if it does not
exist.}
}
\value{
An object of class \code{h5} with methods for interacting with the file.
}
\description{
Creates a file handle that provides a convenient, object-oriented interface
for interacting with and navigating a specific HDF5 file.
}
\details{
This function returns a special \code{h5} object that wraps the standard \code{h5lite}
functions. The primary benefit is that the \code{file} argument is pre-filled,
allowing for more concise and readable code when performing multiple
operations on the same file.

For example, instead of writing:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{h5_write(1:10, file, "dset1")
h5_write(2:20, file, "dset2")
h5_ls(file)
}\if{html}{\out{</div>}}

You can create a handle and use its methods. Note that the \code{file} argument
is omitted from the method calls:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{h5 <- h5_open("my_file.h5")
h5$write(1:10, "dset1")
h5$write(2:20, "dset2")
h5$ls()
h5$close()
}\if{html}{\out{</div>}}
}
\section{Pass-by-Reference Behavior}{

Unlike most R objects, the \code{h5} handle is an \strong{environment}. This means it
is passed by reference. If you assign it to another variable (e.g.,
\code{h5_alias <- h5}), both variables point to the \emph{same} handle. Modifying one
(e.g., by calling \code{h5_alias$close()}) will also affect the other.
}

\section{Interacting with the HDF5 File}{

The \code{h5} object provides several ways to interact with the HDF5 file:

\subsection{Standard \code{h5lite} Functions as Methods}{
Most \code{h5lite} functions (e.g., \code{h5_read}, \code{h5_write}, \code{h5_ls}) are
available as methods on the \code{h5} object, without the \code{h5_} prefix.

For example, \code{h5$write(data, "dset")} is equivalent to
\code{h5_write(data, file, "dset")}.

The available methods are: \code{attr_names}, \code{cd}, \code{class}, \code{close},
\code{create_group}, \code{delete}, \code{dim}, \code{exists}, \code{is_dataset}, \code{is_group},
\code{length}, \code{ls}, \code{move}, \code{names}, \code{pwd}, \code{read}, \code{str}, \code{typeof}, \code{write}.
}

\subsection{Navigation (\verb{$cd()}, \verb{$pwd()})}{
The handle maintains an internal working directory to simplify
path management.
\itemize{
\item{\code{h5$cd(group)}: Changes the handle's internal working directory.
This is a stateful, pass-by-reference operation. It understands absolute
paths (e.g., \code{"/new/path"}) and relative navigation (e.g., \code{"../other"}).
The target group does not need to exist.
}
\item{\code{h5$pwd()}: Returns the current working directory.}
}
When you call a method like \code{h5$read("dset")}, the handle automatically
prepends the current working directory to any relative path. If you provide
an absolute path (e.g., \code{h5$read("/path/to/dset")}), the working directory
is ignored.
}

\subsection{Closing the Handle (\verb{$close()})}{
The \code{h5lite} package does not keep files persistently open. Each operation
opens, modifies, and closes the file. Therefore, the \code{h5$close()} method
does not perform any action on the HDF5 file itself.

Its purpose is to invalidate the handle, preventing any further operations
from being called. After \code{h5$close()} is called, any subsequent method
call (e.g., \code{h5$ls()}) will throw an error.
}
}

\examples{
file <- tempfile(fileext = ".h5")

# Open the handle
h5 <- h5_open(file)

# Write data (note: 'data' is the first argument, 'file' is implicit)
h5$write(1:5, "vector")
h5$write(matrix(1:9, 3, 3), "matrix")

# Create a group and navigate to it
h5$create_group("simulations")
h5$cd("simulations")
print(h5$pwd()) # "/simulations"

# Write data relative to the current working directory
h5$write(rnorm(10), "run1") # Writes to /simulations/run1

# Read data
dat <- h5$read("run1")

# List contents of current WD
h5$ls()

# Close the handle
h5$close()
unlink(file)
}
