/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

typedef herr_t (*H5EA__unprotect_func_t)(void *thing, unsigned cache_flags);

static herr_t  H5EA__lookup_elmt(const H5EA_t *ea, hsize_t idx, bool will_extend, unsigned thing_acc,
                                 void **thing, uint8_t **thing_elmt_buf, hsize_t *thing_elmt_idx,
                                 H5EA__unprotect_func_t *thing_unprot_func);
static H5EA_t *H5EA__new(H5F_t *f, haddr_t ea_addr, bool from_open, void *ctx_udata);

bool H5_PKG_INIT_VAR = false;

const H5EA_class_t *const H5EA_client_class_g[] = {
    H5EA_CLS_CHUNK,      
    H5EA_CLS_FILT_CHUNK, 
    H5EA_CLS_TEST,       
};

H5FL_DEFINE_STATIC(H5EA_t);

H5FL_BLK_DEFINE_STATIC(ea_native_elmt);

static H5EA_t *
H5EA__new(H5F_t *f, haddr_t ea_addr, bool from_open, void *ctx_udata)
{
    H5EA_t     *ea        = NULL; 
    H5EA_hdr_t *hdr       = NULL; 
    H5EA_t     *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(ea_addr));

    
    if (NULL == (ea = H5FL_CALLOC(H5EA_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL, "memory allocation failed for extensible array info");

    
    if (NULL == (hdr = H5EA__hdr_protect(f, ea_addr, ctx_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL, "unable to load extensible array header");

    
    if (from_open && hdr->pending_delete)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTOPENOBJ, NULL, "can't open extensible array pending deletion");

    
    ea->hdr = hdr;
    if (H5EA__hdr_incr(ea->hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");

    
    if (H5EA__hdr_fuse_incr(ea->hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared array header");

    
    ea->f = f;

    
    ret_value = ea;

done:

    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, NULL, "unable to release extensible array header");
    if (!ret_value)
        if (ea && H5EA_close(ea) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CLOSEERROR, NULL, "unable to close extensible array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_t *
H5EA_create(H5F_t *f, const H5EA_create_t *cparam, void *ctx_udata)
{
    H5EA_t *ea = NULL; 
    haddr_t ea_addr;   
    H5EA_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(cparam);

    
    HDcompile_assert(H5EA_NUM_CLS_ID == NELMTS(H5EA_client_class_g));

    
    if (HADDR_UNDEF == (ea_addr = H5EA__hdr_create(f, cparam, ctx_udata)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, NULL, "can't create extensible array header");

    
    if (NULL == (ea = H5EA__new(f, ea_addr, false, ctx_udata)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, NULL,
                    "allocation and/or initialization failed for extensible array wrapper");

    
    ret_value = ea;

done:
    if (!ret_value)
        if (ea && H5EA_close(ea) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CLOSEERROR, NULL, "unable to close extensible array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_t *
H5EA_open(H5F_t *f, haddr_t ea_addr, void *ctx_udata)
{
    H5EA_t *ea        = NULL; 
    H5EA_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(H5_addr_defined(ea_addr));

    
    if (NULL == (ea = H5EA__new(f, ea_addr, true, ctx_udata)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, NULL,
                    "allocation and/or initialization failed for extensible array wrapper");

    
    ret_value = ea;

done:
    if (!ret_value)
        if (ea && H5EA_close(ea) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CLOSEERROR, NULL, "unable to close extensible array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_get_nelmts(const H5EA_t *ea, hsize_t *nelmts)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(ea);
    assert(nelmts);

    
    *nelmts = ea->hdr->stats.stored.max_idx_set;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5EA_get_addr(const H5EA_t *ea, haddr_t *addr)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(ea);
    assert(ea->hdr);
    assert(addr);

    
    *addr = ea->hdr->addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__lookup_elmt(const H5EA_t *ea, hsize_t idx, bool will_extend, unsigned thing_acc, void **thing,
                  uint8_t **thing_elmt_buf, hsize_t *thing_elmt_idx,
                  H5EA__unprotect_func_t *thing_unprot_func)
{
    H5EA_hdr_t       *hdr                = ea->hdr;            
    H5EA_iblock_t    *iblock             = NULL;               
    H5EA_sblock_t    *sblock             = NULL;               
    H5EA_dblock_t    *dblock             = NULL;               
    H5EA_dblk_page_t *dblk_page          = NULL;               
    unsigned          iblock_cache_flags = H5AC__NO_FLAGS_SET; 
    unsigned          sblock_cache_flags = H5AC__NO_FLAGS_SET; 
    bool              stats_changed      = false;              
    bool              hdr_dirty          = false;              
    herr_t            ret_value          = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(ea);
    assert(hdr);
    assert(thing);
    assert(thing_elmt_buf);
    assert(thing_unprot_func);

    
    assert((thing_acc & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    hdr->f = ea->f;

    
    *thing             = NULL;
    *thing_elmt_buf    = NULL;
    *thing_elmt_idx    = 0;
    *thing_unprot_func = (H5EA__unprotect_func_t)NULL;

    
    if (!H5_addr_defined(hdr->idx_blk_addr)) {
        
        if (0 == (thing_acc & H5AC__READ_ONLY_FLAG)) { 
            
            hdr->idx_blk_addr = H5EA__iblock_create(hdr, &stats_changed);
            if (!H5_addr_defined(hdr->idx_blk_addr))
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL, "unable to create index block");
            hdr_dirty = true;
        } 
        else
            HGOTO_DONE(SUCCEED);
    } 

    
    if (NULL == (iblock = H5EA__iblock_protect(hdr, thing_acc)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array index block, address = %llu",
                    (unsigned long long)hdr->idx_blk_addr);

    
    if (idx < hdr->cparam.idx_blk_elmts) {
        
        *thing             = iblock;
        *thing_elmt_buf    = (uint8_t *)iblock->elmts;
        *thing_elmt_idx    = idx;
        *thing_unprot_func = (H5EA__unprotect_func_t)H5EA__iblock_unprotect;
    } 
    else {
        unsigned sblk_idx; 
        size_t   dblk_idx; 
        hsize_t  elmt_idx; 

        
        sblk_idx = H5EA__dblock_sblk_idx(hdr, idx);

        
        elmt_idx = idx - (hdr->cparam.idx_blk_elmts + hdr->sblk_info[sblk_idx].start_idx);

        
        if (sblk_idx < iblock->nsblks) {
            
            dblk_idx = (size_t)(hdr->sblk_info[sblk_idx].start_dblk +
                                (elmt_idx / hdr->sblk_info[sblk_idx].dblk_nelmts));
            assert(dblk_idx < iblock->ndblk_addrs);

            
            if (!H5_addr_defined(iblock->dblk_addrs[dblk_idx])) {
                
                if (0 == (thing_acc & H5AC__READ_ONLY_FLAG)) { 
                    haddr_t dblk_addr;                         
                    hsize_t dblk_off;                          

                    
                    dblk_off = hdr->sblk_info[sblk_idx].start_idx +
                               (dblk_idx * hdr->sblk_info[sblk_idx].dblk_nelmts);
                    dblk_addr = H5EA__dblock_create(hdr, iblock, &stats_changed, dblk_off,
                                                    hdr->sblk_info[sblk_idx].dblk_nelmts);
                    if (!H5_addr_defined(dblk_addr))
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL,
                                    "unable to create extensible array data block");

                    
                    iblock->dblk_addrs[dblk_idx] = dblk_addr;
                    iblock_cache_flags |= H5AC__DIRTIED_FLAG;
                } 
                else
                    HGOTO_DONE(SUCCEED);
            } 

            
            if (NULL == (dblock = H5EA__dblock_protect(hdr, iblock, iblock->dblk_addrs[dblk_idx],
                                                       hdr->sblk_info[sblk_idx].dblk_nelmts, thing_acc)))
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                            "unable to protect extensible array data block, address = %llu",
                            (unsigned long long)iblock->dblk_addrs[dblk_idx]);

            
            elmt_idx %= hdr->sblk_info[sblk_idx].dblk_nelmts;

            
            if (will_extend && !dblock->has_hdr_depend) {
                if (H5EA__create_flush_depend((H5AC_info_t *)hdr, (H5AC_info_t *)dblock) < 0)
                    HGOTO_ERROR(
                        H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                        "unable to create flush dependency between data block and header, index = %llu",
                        (unsigned long long)idx);
                dblock->has_hdr_depend = true;
            } 

            
            *thing             = dblock;
            *thing_elmt_buf    = (uint8_t *)dblock->elmts;
            *thing_elmt_idx    = elmt_idx;
            *thing_unprot_func = (H5EA__unprotect_func_t)H5EA__dblock_unprotect;
        } 
        else {
            size_t sblk_off; 

            
            sblk_off = sblk_idx - iblock->nsblks;

            
            if (!H5_addr_defined(iblock->sblk_addrs[sblk_off])) {
                
                if (0 == (thing_acc & H5AC__READ_ONLY_FLAG)) { 
                    haddr_t sblk_addr;                         

                    
                    sblk_addr = H5EA__sblock_create(hdr, iblock, &stats_changed, sblk_idx);
                    if (!H5_addr_defined(sblk_addr))
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL,
                                    "unable to create extensible array super block");

                    
                    iblock->sblk_addrs[sblk_off] = sblk_addr;
                    iblock_cache_flags |= H5AC__DIRTIED_FLAG;
                } 
                else
                    HGOTO_DONE(SUCCEED);
            } 

            
            if (NULL == (sblock = H5EA__sblock_protect(hdr, iblock, iblock->sblk_addrs[sblk_off], sblk_idx,
                                                       thing_acc)))
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                            "unable to protect extensible array super block, address = %llu",
                            (unsigned long long)iblock->sblk_addrs[sblk_off]);

            
            dblk_idx = (size_t)(elmt_idx / sblock->dblk_nelmts);
            assert(dblk_idx < sblock->ndblks);

            
            if (!H5_addr_defined(sblock->dblk_addrs[dblk_idx])) {
                
                if (0 == (thing_acc & H5AC__READ_ONLY_FLAG)) { 
                    haddr_t dblk_addr;                         
                    hsize_t dblk_off;                          

                    
                    dblk_off = hdr->sblk_info[sblk_idx].start_idx +
                               (dblk_idx * hdr->sblk_info[sblk_idx].dblk_nelmts);
                    dblk_addr =
                        H5EA__dblock_create(hdr, sblock, &stats_changed, dblk_off, sblock->dblk_nelmts);
                    if (!H5_addr_defined(dblk_addr))
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL,
                                    "unable to create extensible array data block");

                    
                    sblock->dblk_addrs[dblk_idx] = dblk_addr;
                    sblock_cache_flags |= H5AC__DIRTIED_FLAG;

                    
                    if (will_extend && !sblock->has_hdr_depend) {
                        if (H5EA__create_flush_depend((H5AC_info_t *)sblock->hdr, (H5AC_info_t *)sblock) < 0)
                            HGOTO_ERROR(
                                H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                                "unable to create flush dependency between super block and header, address "
                                "= %llu",
                                (unsigned long long)sblock->addr);
                        sblock->has_hdr_depend = true;
                    } 
                }     
                else
                    HGOTO_DONE(SUCCEED);
            } 

            
            elmt_idx %= sblock->dblk_nelmts;

            
            if (sblock->dblk_npages) {
                haddr_t dblk_page_addr; 
                size_t  page_idx;       
                size_t  page_init_idx;  

                
                page_idx = (size_t)elmt_idx / hdr->dblk_page_nelmts;

                
                page_init_idx = (dblk_idx * sblock->dblk_npages) + page_idx;

                
                elmt_idx %= hdr->dblk_page_nelmts;

                
                dblk_page_addr = sblock->dblk_addrs[dblk_idx] + H5EA_DBLOCK_PREFIX_SIZE(sblock) +
                                 (page_idx * sblock->dblk_page_size);

                
                if (!H5VM_bit_get(sblock->page_init, page_init_idx)) {
                    
                    if (0 == (thing_acc & H5AC__READ_ONLY_FLAG)) { 
                        
                        if (H5EA__dblk_page_create(hdr, sblock, dblk_page_addr) < 0)
                            HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL, "unable to create data block page");

                        
                        H5VM_bit_set(sblock->page_init, page_init_idx, true);
                        sblock_cache_flags |= H5AC__DIRTIED_FLAG;
                    } 
                    else
                        HGOTO_DONE(SUCCEED);
                } 

                
                if (NULL == (dblk_page = H5EA__dblk_page_protect(hdr, sblock, dblk_page_addr, thing_acc)))
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                                "unable to protect extensible array data block page, address = %llu",
                                (unsigned long long)dblk_page_addr);

                
                if (will_extend && !dblk_page->has_hdr_depend) {
                    if (H5EA__create_flush_depend((H5AC_info_t *)hdr, (H5AC_info_t *)dblk_page) < 0)
                        HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                                    "unable to create flush dependency between data block page and header, "
                                    "index = %llu",
                                    (unsigned long long)idx);
                    dblk_page->has_hdr_depend = true;
                } 

                
                *thing             = dblk_page;
                *thing_elmt_buf    = (uint8_t *)dblk_page->elmts;
                *thing_elmt_idx    = elmt_idx;
                *thing_unprot_func = (H5EA__unprotect_func_t)H5EA__dblk_page_unprotect;
            } 
            else {
                
                if (NULL == (dblock = H5EA__dblock_protect(hdr, sblock, sblock->dblk_addrs[dblk_idx],
                                                           sblock->dblk_nelmts, thing_acc)))
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                                "unable to protect extensible array data block, address = %llu",
                                (unsigned long long)sblock->dblk_addrs[dblk_idx]);

                
                if (will_extend && !dblock->has_hdr_depend) {
                    if (H5EA__create_flush_depend((H5AC_info_t *)hdr, (H5AC_info_t *)dblock) < 0)
                        HGOTO_ERROR(
                            H5E_EARRAY, H5E_CANTDEPEND, FAIL,
                            "unable to create flush dependency between data block and header, index = %llu",
                            (unsigned long long)idx);
                    dblock->has_hdr_depend = true;
                } 

                
                *thing             = dblock;
                *thing_elmt_buf    = (uint8_t *)dblock->elmts;
                *thing_elmt_idx    = elmt_idx;
                *thing_unprot_func = (H5EA__unprotect_func_t)H5EA__dblock_unprotect;
            } 
        }     
    }         

    
    assert(*thing != NULL);
    assert(*thing_unprot_func != NULL);

done:
    
    if (ret_value < 0) {
        *thing             = NULL;
        *thing_elmt_buf    = NULL;
        *thing_elmt_idx    = 0;
        *thing_unprot_func = (H5EA__unprotect_func_t)NULL;
    } 

    
    if (stats_changed)
        hdr_dirty = true;

    
    if (hdr_dirty)
        if (H5EA__hdr_modified(hdr) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTMARKDIRTY, FAIL,
                        "unable to mark extensible array header as modified");

    
    if (iblock && *thing != iblock && H5EA__iblock_unprotect(iblock, iblock_cache_flags) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array index block");
    
    if (sblock && H5EA__sblock_unprotect(sblock, sblock_cache_flags) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array super block");
    if (dblock && *thing != dblock && H5EA__dblock_unprotect(dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array data block");
    if (dblk_page && *thing != dblk_page && H5EA__dblk_page_unprotect(dblk_page, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to release extensible array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_set(const H5EA_t *ea, hsize_t idx, const void *elmt)
{
    H5EA_hdr_t *hdr = ea->hdr; 
    void    *thing = NULL; 
    uint8_t *thing_elmt_buf; 
    hsize_t  thing_elmt_idx; 
    H5EA__unprotect_func_t thing_unprot_func; 
    bool                   will_extend; 
    unsigned               thing_cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t                 ret_value         = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(ea);
    assert(hdr);

    
    hdr->f = ea->f;

    
    will_extend = (idx >= hdr->stats.stored.max_idx_set);
    if (H5EA__lookup_elmt(ea, idx, will_extend, H5AC__NO_FLAGS_SET, &thing, &thing_elmt_buf, &thing_elmt_idx,
                          &thing_unprot_func) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to protect array metadata");

    
    assert(thing);
    assert(thing_elmt_buf);
    assert(thing_unprot_func);

    
    H5MM_memcpy(thing_elmt_buf + (hdr->cparam.cls->nat_elmt_size * thing_elmt_idx), elmt,
                hdr->cparam.cls->nat_elmt_size);
    thing_cache_flags |= H5AC__DIRTIED_FLAG;

    
    if (will_extend) {
        
        hdr->stats.stored.max_idx_set = idx + 1;
        if (H5EA__hdr_modified(hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTMARKDIRTY, FAIL,
                        "unable to mark extensible array header as modified");
    }

done:
    
    if (thing && (thing_unprot_func)(thing, thing_cache_flags) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array metadata");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_get(const H5EA_t *ea, hsize_t idx, void *elmt)
{
    H5EA_hdr_t *hdr = ea->hdr; 
    void *thing = NULL; 
    H5EA__unprotect_func_t thing_unprot_func =
        NULL; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(ea);
    assert(hdr);

    
    if (idx >= hdr->stats.stored.max_idx_set) {
        
        if ((hdr->cparam.cls->fill)(elmt, (size_t)1) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, FAIL, "can't set element to class's fill value");
    } 
    else {
        uint8_t *thing_elmt_buf; 
        hsize_t  thing_elmt_idx; 

        
        hdr->f = ea->f;

        
        if (H5EA__lookup_elmt(ea, idx, false, H5AC__READ_ONLY_FLAG, &thing, &thing_elmt_buf, &thing_elmt_idx,
                              &thing_unprot_func) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to protect array metadata");

        
        if (NULL == thing) {
            
            if ((hdr->cparam.cls->fill)(elmt, (size_t)1) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, FAIL, "can't set element to class's fill value");
        } 
        else
            
            H5MM_memcpy(elmt, thing_elmt_buf + (hdr->cparam.cls->nat_elmt_size * thing_elmt_idx),
                        hdr->cparam.cls->nat_elmt_size);
    } 

done:
    
    if (thing && (thing_unprot_func)(thing, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array metadata");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_depend(H5EA_t *ea, H5AC_proxy_entry_t *parent)
{
    H5EA_hdr_t *hdr       = ea->hdr; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(ea);
    assert(hdr);
    assert(parent);

    
    if (NULL == hdr->parent) {
        
        assert(hdr->top_proxy);

        
        hdr->f = ea->f;

        
        if (H5AC_proxy_entry_add_child(parent, hdr->f, hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, FAIL, "unable to add extensible array as child of proxy");
        hdr->parent = parent;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_close(H5EA_t *ea)
{
    bool    pending_delete = false;       
    haddr_t ea_addr        = HADDR_UNDEF; 
    herr_t  ret_value      = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(ea);

    
    if (ea->hdr) {
        
        if (0 == H5EA__hdr_fuse_decr(ea->hdr)) {
            
            ea->hdr->f = ea->f;

            

            
            if (ea->hdr->pending_delete) {
                
                pending_delete = true;
                ea_addr        = ea->hdr->addr;
            } 
        }     

        
        if (pending_delete) {
            H5EA_hdr_t *hdr; 

#ifndef NDEBUG
            {
                unsigned hdr_status = 0; 

                
                if (H5AC_get_entry_status(ea->f, ea_addr, &hdr_status) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL,
                                "unable to check metadata cache status for extensible array header");

                
                assert(hdr_status & H5AC_ES__IN_CACHE);
                assert(hdr_status & H5AC_ES__IS_PINNED);
                assert(!(hdr_status & H5AC_ES__IS_PROTECTED));
            }
#endif 

            
            
            if (NULL == (hdr = H5EA__hdr_protect(ea->f, ea_addr, NULL, H5AC__NO_FLAGS_SET)))
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTLOAD, FAIL, "unable to load extensible array header");

            
            hdr->f = ea->f;

            
            
            if (H5EA__hdr_decr(ea->hdr) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                            "can't decrement reference count on shared array header");

            
            if (H5EA__hdr_delete(hdr) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL, "unable to delete extensible array");
        } 
        else {
            
            
            if (H5EA__hdr_decr(ea->hdr) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                            "can't decrement reference count on shared array header");
        } 
    }     

    
    ea = (H5EA_t *)H5FL_FREE(H5EA_t, ea);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_delete(H5F_t *f, haddr_t ea_addr, void *ctx_udata)
{
    H5EA_hdr_t *hdr       = NULL; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(ea_addr));

    
    if (NULL == (hdr = H5EA__hdr_protect(f, ea_addr, ctx_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array header, address = %llu", (unsigned long long)ea_addr);

    
    if (hdr->file_rc)
        hdr->pending_delete = true;
    else {
        
        hdr->f = f;

        
        if (H5EA__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL, "unable to delete extensible array");
        hdr = NULL;
    }

done:
    
    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5EA_iterate(H5EA_t *ea, H5EA_operator_t op, void *udata)
{
    uint8_t *elmt = NULL;
    hsize_t  u;
    int      ret_value = H5_ITER_CONT;

    FUNC_ENTER_NOAPI(H5_ITER_ERROR)

    
    assert(ea);
    assert(op);
    assert(udata);

    
    if (NULL == (elmt = H5FL_BLK_MALLOC(ea_native_elmt, ea->hdr->cparam.cls->nat_elmt_size)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, H5_ITER_ERROR,
                    "memory allocation failed for extensible array element");

    
    for (u = 0; u < ea->hdr->stats.stored.max_idx_set && ret_value == H5_ITER_CONT; u++) {
        
        if (H5EA_get(ea, u, elmt) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, H5_ITER_ERROR, "unable to delete fixed array");

        
        if ((ret_value = (*op)(u, elmt, udata)) < 0) {
            HERROR(H5E_EARRAY, H5E_BADITER, "iteration callback error");
            break;
        }
    }

done:
    if (elmt)
        elmt = H5FL_BLK_FREE(ea_native_elmt, elmt);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA_patch_file(H5EA_t *ea, H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(ea);
    assert(f);

    if (ea->f != f || ea->hdr->f != f)
        ea->f = ea->hdr->f = f;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
