% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ibdsim.R
\name{ibdsim}
\alias{ibdsim}
\title{IBD simulation}
\usage{
ibdsim(
  x,
  N = 1,
  ids = NULL,
  map = "decode",
  model = c("chi", "haldane"),
  skipRecomb = NULL,
  simplify1 = TRUE,
  seed = NULL,
  verbose = TRUE
)
}
\arguments{
\item{x}{A \code{\link[pedtools:ped]{pedtools::ped()}} object.}

\item{N}{A positive integer indicating the number of simulations.}

\item{ids}{A vector of names indicating which pedigree members should be
included in the output. Alternatively, a function taking \code{x} as input and
returning such a vector, like \code{\link[pedtools:ped_subgroups]{pedtools::leaves()}}. By default (\code{ids = NULL}) everyone is included.}

\item{map}{The genetic map to be used in the simulations: Allowed values are:
\itemize{
\item a \code{genomeMap} object, typically produced by \code{\link[=loadMap]{loadMap()}}
\item a single \code{chromMap} object, for instance as produced by \code{\link[=uniformMap]{uniformMap()}}
\item a character, which is passed on to \code{\link[=loadMap]{loadMap()}} with default parameters.
Currently the only valid option is "decode19" (or abbreviations of this).
}

Default: "decode19".}

\item{model}{Either "chi" or "haldane", indicating the statistical model for
recombination (see Details). Default: "chi".}

\item{skipRecomb}{A vector of ID labels indicating individuals whose meioses
should be simulated without recombination. (Each child will then receive a
random strand of each chromosome.) Alternatively, a function taking \code{x} as
input, e.g., \code{\link[pedtools:ped_subgroups]{pedtools::founders()}}. By default, recombination is skipped
in founders who are uninformative for IBD sharing in the \code{ids} individuals.}

\item{simplify1}{A logical, by default TRUE, removing the outer list layer
when N = 1. See Value.}

\item{seed}{An integer to be passed on to \code{\link[=set.seed]{set.seed()}}).}

\item{verbose}{A logical.}
}
\value{
If \code{N > 1}, a list of \code{N} objects of class \code{genomeSim}.

If \code{N = 1}, the outer list layer is removed by default, which is typically
desired in interactive use and in pipe chains. To enforce a list output in
this case, add \code{simplify1 = FALSE}.

A \code{genomeSim} object is essentially a numerical matrix describing the
allele flow through the pedigree in a single genome simulation. Each row
corresponds to a chromosomal segment. The first 3 columns (chrom, startMB,
endMB) describe the physical megabase location of the segment. Next come
the centimorgan coordinates (startCM, endCM), which are computed from \code{map}
by averaging the male and female values. Then follow the allele columns,
two for each individual in \code{ids}, suffixed by ":p" and ":m" signifying the
paternal and maternal alleles, respectively.

If \code{ids} has length 1, a column named \code{Aut} is added, whose entries are 1
for autozygous segments and 0 otherwise.

If \code{ids} has length 2, two columns are added:
\itemize{
\item \code{IBD} : The IBD status of each segment, i.e., the number of alleles shared
identical by descent. This is always either 0, 1, 2 or NA, where the latter
appear if either individual (or both) of individuals is autozygous in the
segment.
\item \code{Sigma} : The condensed identity ("Jacquard") state of each segment,
given as an integer in the range 1-9. The numbers correspond to the
standard ordering of the condensed states. In particular, for non-inbred
individuals, the states 9, 8, 7 correspond to IBD status 0, 1, 2
respectively.
}
}
\description{
This is the main function of the package, simulating the recombination
process in each meiosis of a pedigree. The output summarises the IBD segments
between all or a subset of individuals.
}
\details{
Each simulation starts by distributing unique alleles (labelled 1, 2, ...) to
the pedigree founders, followed by a separate recombination process for each
chromosome, according to the value of \code{model}:
\itemize{
\item \code{model = "chi"} (default): This uses a renewal process along the
four-strand bundle, with waiting times following a chi-square distribution.
\item \code{model = "haldane"}: In this model, crossover events are modelled as a
Poisson process along each chromosome. (Faster, but less realistic.)
}

Recombination rates along each chromosome are determined by the \code{map}
parameter. The default value ("decode19") loads a thinned version of the
recombination map of the human genome published by Halldorsson et al (2019).

In many applications, the fine-scale default map is not necessary, and may be
replaced by simpler maps with constant recombination rates. See
\code{\link[=uniformMap]{uniformMap()}} and \code{\link[=loadMap]{loadMap()}} for ways to produce such maps.
}
\examples{

### Example 1: Half siblings ###

hs = halfSibPed()
sim = ibdsim(hs, map = uniformMap(M = 1), seed = 10)
sim

# Plot haplotypes
haploDraw(hs, sim)

#' ### Example 2: Full sib mating ###

x = fullSibMating(1)
sim = ibdsim(x, ids = 5:6, map = uniformMap(M = 10), seed = 1)
head(sim)

# All 9 identity states are present
stopifnot(setequal(sim[, 'Sigma'], 1:9))

}
\references{
Halldorsson et al. \emph{Characterizing mutagenic effects of
recombination through a sequence-level genetic map.} Science 363, no. 6425
(2019).
}
