% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write.R
\name{write_tif}
\alias{write_tif}
\alias{tif_write}
\title{Write images in TIFF format}
\usage{
write_tif(
  img,
  path,
  bits_per_sample = "auto",
  compression = "none",
  overwrite = FALSE,
  msg = TRUE,
  tags_to_write = NULL
)

tif_write(
  img,
  path,
  bits_per_sample = "auto",
  compression = "none",
  overwrite = FALSE,
  msg = TRUE,
  tags_to_write = NULL
)
}
\arguments{
\item{img}{An array representing the image. \itemize{\item For a
single-plane, grayscale image, use a matrix \code{img[y, x]}. \item For a
multi-plane, grayscale image, use a 3-dimensional array \code{img[y, x, plane]}.
\item For a multi-channel, single-plane image, use a 4-dimensional array
with a redundant 4th slot \code{img[y, x, channel, ]} (see \link{ijtiff_img}
'Examples' for an example). \item For a multi-channel, multi-plane image,
use a 4-dimensional array \code{img[y, x, channel, plane]}.}}

\item{path}{Path to the TIFF file to write to.}

\item{bits_per_sample}{Number of bits per sample (numeric scalar). Supported
values are 8, 16, and 32. The default \code{"auto"} automatically picks the
smallest workable value based on the maximum element in \code{img}. For example,
if the maximum element in \code{img} is 789, then 16-bit will be chosen because
789 is greater than 2 ^ 8 - 1 but less than or equal to 2 ^ 16 - 1.}

\item{compression}{A string, the desired compression algorithm. Must be one
of \code{"none"}, \code{"LZW"}, \code{"PackBits"}, \code{"RLE"}, \code{"JPEG"}, \code{"deflate"} or
\code{"Zip"}. If you want compression but don't know which one to go for, I
recommend \code{"Zip"}, it gives a large file size reduction and it's lossless.
Note that \code{"deflate"} and \code{"Zip"} are the same thing. Avoid using \code{"JPEG"}
compression in a TIFF file if you can; I've noticed it can be buggy.}

\item{overwrite}{If writing the image would overwrite a file, do you want to
proceed?}

\item{msg}{Print an informative message about the image being written?}

\item{tags_to_write}{A named list of TIFF tags to write. Tag names are
case-insensitive and hyphens/underscores are ignored (e.g., "X_Resolution",
"x-resolution", and "xresolution" are all equivalent).

If \code{NULL} (the default), any supported TIFF tags present as attributes on
\code{img} (e.g., from \code{read_tif()}) will be automatically extracted and written.
If \code{tags_to_write} is specified, it is merged with auto-detected tags, with
explicit tags taking precedence. This enables round-trip editing where
metadata from a read image is preserved on re-export unless explicitly
overridden.

Supported tags are:
\itemize{
\item \code{xresolution} - Numeric value for horizontal resolution in pixels per
unit
\item \code{yresolution} - Numeric value for vertical resolution in pixels per unit
\item \code{resolutionunit} - Integer: 1 (none), 2 (inch), or 3 (centimeter)
\item \code{orientation} - Integer 1-8 for image orientation
\item \code{xposition} - Numeric value for horizontal position in resolution units
\item \code{yposition} - Numeric value for vertical position in resolution units
\item \code{copyright} - Character string for copyright notice
\item \code{artist} - Character string for creator name
\item \code{documentname} - Character string for document name
\item \code{datetime} - Date/time (character, Date, or POSIXct)
\item \code{imagedescription} - Character string for image description
}}
}
\value{
The input \code{img} (invisibly).
}
\description{
Write images into a TIFF file.
}
\examples{
img <- read_tif(system.file("img", "Rlogo.tif", package = "ijtiff"))
temp_dir <- tempdir()

# Basic write
write_tif(img, paste0(temp_dir, "/", "Rlogo"))

# Round-trip with automatic metadata preservation
# Metadata from read_tif is automatically preserved
img_with_metadata <- read_tif(system.file("img", "Rlogo.tif", 
                                          package = "ijtiff"))
write_tif(img_with_metadata, paste0(temp_dir, "/", "Rlogo_preserved"))

# Write with additional/overridden tags
write_tif(img_with_metadata, paste0(temp_dir, "/", "Rlogo_with_tags"),
          tags_to_write = list(
            artist = "R Core Team",
            copyright = "(c) 2024",
            imagedescription = "The R logo"
          ))

img <- matrix(1:4, nrow = 2)
write_tif(img, paste0(temp_dir, "/", "tiny2x2"))
list.files(temp_dir, pattern = "tif$")
}
\seealso{
\code{\link[=read_tif]{read_tif()}}
}
\author{
Simon Urbanek wrote most of this code for the 'tiff' package. Rory
Nolan lifted it from there and changed it around a bit for this 'ijtiff'
package. Credit should be directed towards Lord Urbanek.
}
