#' Dynamic Prediction of Transition Probabilities
#'
#' @description
#' Computes personalized dynamic predictions of transition probabilities
#' from a fitted \code{jmSurface} model. Given a patient's biomarker
#' history up to a landmark time, projects the latent trajectories forward
#' and integrates the transition-specific hazard to obtain cumulative
#' transition probabilities.
#'
#' @param object A \code{"jmSurface"} object from \code{jmSurf}.
#' @param patient_id Integer patient identifier.
#' @param landmark Numeric landmark time (predict from this time).
#' @param horizon Numeric prediction horizon (predict this many years ahead).
#' @param n_points Integer number of time points for the prediction grid.
#'   Default \code{60}.
#'
#' @return Data frame with columns:
#'   \item{time}{Absolute time points}
#'   \item{risk}{Cumulative transition probability}
#'   \item{hazard}{Instantaneous hazard at each time point}
#'   \item{transition}{Transition name}
#'   \item{to_state}{Target state}
#'   \item{patient_id}{Patient identifier}
#'   \item{landmark}{Landmark time used}
#'
#' @details
#' The conditional transition probability is computed as:
#' \deqn{\pi_i^{rs}(t_L, \Delta t) = 1 - \exp\left\{-\int_{t_L}^{t_L+\Delta t} \lambda_i^{rs}(u | \hat\eta_i(u)) du\right\}}
#' where \eqn{\hat\eta_i(u)} is the BLUP-projected trajectory and the integral
#' is approximated via the Breslow estimator.
#'
#' @export
dynPred <- function(object, patient_id, landmark = 0, horizon = 3,
                    n_points = 60) {

  if (!inherits(object, "jmSurface"))
    stop("object must be of class 'jmSurface'")

  ## Normalize arrows in surv_data
  surv <- object$surv_data
  surv$transition <- gsub("\u2192", "->", surv$transition)
  ps <- surv[surv$patient_id == patient_id, ]
  if (nrow(ps) == 0) stop("Patient ", patient_id, " not found in surv_data.")

  ## Determine current state at landmark time
  ps <- ps[order(ps$stop_time), ]
  current_state <- ps$state_from[1]  # default

  for (i in seq_len(nrow(ps))) {
    if (as.numeric(ps$stop_time[i]) <= landmark && ps$status[i] == 1) {
      current_state <- ps$state_to[i]
    }
  }

  ## Find transitions from current state (handle both arrow styles)
  possible_tr <- object$transitions[
    grepl(paste0("^", current_state, " -> "), object$transitions)
  ]

  if (length(possible_tr) == 0)
    stop("No fitted transitions from state '", current_state, "'")

  all_results <- list()

  for (tr in possible_tr) {
    tr_parts <- strsplit(tr, " -> ")[[1]]
    to_state <- trimws(tr_parts[2])

    gf <- object$gam_fits[[tr]]
    ed <- object$eta_data[[tr]]
    if (is.null(gf)) next

    eta_cols <- grep("^eta_", names(ed), value = TRUE)
    t_rel <- seq(0.05, horizon, length.out = n_points)
    t_abs <- landmark + t_rel

    ## Build prediction data frame
    pred_df <- data.frame(time_abs = t_abs)

    ## Project BLUP trajectories
    for (mk in names(object$lme_fits)) {
      if (is.null(object$lme_fits[[mk]])) next
      cm <- coef(object$lme_fits[[mk]])
      mk_clean <- gsub("[^A-Za-z0-9]", "", mk)
      pc <- as.character(patient_id)
      eta_name <- paste0("eta_", mk_clean)
      if (pc %in% rownames(cm)) {
        pred_df[[eta_name]] <- cm[pc, 1] + cm[pc, 2] * t_abs
      } else {
        pred_df[[eta_name]] <- 0
      }
    }

    ## Add covariates
    sr <- surv[surv$patient_id == patient_id, ][1, ]
    for (cv in object$covariates) {
      if (cv %in% names(sr)) {
        pred_df[[cv]] <- as.numeric(sr[[cv]])
      }
    }

    ## Ensure all eta columns present
    for (ec in eta_cols) {
      if (!(ec %in% names(pred_df))) pred_df[[ec]] <- 0
    }

    ## Add required technical columns
    pred_df$time_in_state <- median(ed$time_in_state, na.rm = TRUE)
    pred_df$status <- 1

    ## Compute linear predictor
    lp <- tryCatch(
      predict(gf, newdata = pred_df, type = "link"),
      error = function(e) rep(0, nrow(pred_df))
    )

    ## Breslow-based risk estimation
    lp_train <- predict(gf, type = "link")
    lp_mean <- mean(lp_train, na.rm = TRUE)

    ord <- order(ed$time_in_state)
    t_sorted <- ed$time_in_state[ord]
    status_sorted <- ed$status[ord]
    lp_sorted <- lp_train[ord]

    n_events <- sum(ed$status)
    total_py <- sum(ed$time_in_state)
    lambda0_avg <- n_events / (total_py * mean(exp(lp_sorted), na.rm = TRUE))

    h_t <- lambda0_avg * exp(lp - lp_mean)
    dt <- diff(c(0, t_rel))
    cum_haz <- cumsum(h_t * dt)
    risk <- 1 - exp(-cum_haz)
    risk <- pmin(pmax(risk, 0), 0.999)

    all_results[[tr]] <- data.frame(
      time = t_abs,
      risk = risk,
      hazard = as.numeric(h_t),
      transition = tr,
      to_state = to_state,
      patient_id = patient_id,
      landmark = landmark,
      stringsAsFactors = FALSE
    )
  }

  do.call(rbind, all_results)
}
