% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeMemory.R
\name{computeMemory}
\alias{computeMemory}
\title{Quantifies ecological memory with Random Forest.}
\usage{
computeMemory(
  lagged.data = NULL,
  response = NULL,
  drivers = NULL,
  random.mode = "autocorrelated",
  repetitions = 10,
  subset.response = "none",
  num.threads = 2
)
}
\arguments{
\item{lagged.data}{a lagged dataset resulting from \code{\link{prepareLaggedData}}. See \code{\link{palaeodataLagged}} as example. Default: \code{NULL}.}

\item{response}{character string, name of the response variable. Not required if `lagged.data` was generated with [prepareLaggedData]. Default: \code{NULL}.}

\item{drivers}{a character string or character vector with variables to be used as predictors in the model. Not required if `lagged.data` was generated with [prepareLaggedData]. \strong{Important:} \code{drivers} names must not have the character "__" (double underscore). Default: \code{NULL}.}

\item{random.mode}{either "none", "white.noise" or "autocorrelated". See details. Default: \code{"autocorrelated"}.}

\item{repetitions}{integer, number of random forest models to fit. Default: \code{10}.}

\item{subset.response}{character string with values "up", "down" or "none", triggers the subsetting of the input dataset. "up" only models memory on cases where the response's trend is positive, "down" selects cases with negative trends, and "none" selects all cases. Default: \code{"none"}.}

\item{num.threads}{integer, number of cores \link[ranger]{ranger} can use for multithreading. Default: \code{2}.}
}
\value{
A list with 5 slots:
 \itemize{
 \item \code{response} character, response variable name.
 \item \code{drivers} character vector, driver variable names.
 \item \code{memory} dataframe with six columns:
    \itemize{
      \item \code{median} numeric, median importance across \code{repetitions} of the given \code{variable} according to Random Forest.
      \item \code{sd} numeric, standard deviation of the importance values of the given \code{variable} across \code{repetitions}.
      \item \code{min} and \code{max} numeric, percentiles 0.05 and 0.95 of importance values of the given \code{variable} across \code{repetitions}.
      \item \code{variable} character, names of the different variables used to model ecological memory.
      \item \code{lag} numeric, time lag values.
    }
 \item \code{R2} vector, values of pseudo R-squared value obtained for the Random Forest model fitted on each repetition. Pseudo R-squared is the Pearson correlation between the observed and predicted data.
 \item \code{prediction} dataframe, with the same columns as the dataframe in the slot \code{memory}, with the median and confidence intervals of the predictions of all random forest models fitted.
}
}
\description{
Takes the output of \code{\link{prepareLaggedData}} to fit the following model with Random Forest:

 \eqn{p_{t} = p_{t-1} +...+ p_{t-n} + d_{t} + d_{t-1} +...+ d_{t-n} + r}

 where:

\itemize{
 \item \eqn{d} is a driver (several drivers can be added).
 \item \eqn{t} is the time of any given value of the response \emph{p}.
 \item \eqn{t-1} is the lag number 1 (in time units).
 \item \eqn{p_{t-1} +...+ p_{t-n}}  represents the endogenous component of ecological memory.
 \item \eqn{d_{t-1} +...+ d_{t-n}}  represents the exogenous component of ecological memory.
 \item \eqn{d_{t}} represents the concurrent effect of the driver over the response.
 \item \eqn{r} represents a column of random values, used to test the significance of the variable importance scores returned by Random Forest.
}
}
\details{
This function uses the \link[ranger]{ranger} package to fit Random Forest models. Please, check the help of the \link[ranger]{ranger} function to better understand how Random Forest is parameterized in this package. This function fits the model explained above as many times as defined in the argument \code{repetitions}.

To test the statistical significance of the variable importance scores returned by random forest, on each repetition the model is fitted with a different \code{r} (random) term, unless \code{random.mode = "none"}. If \code{random.mode} equals "autocorrelated", the random term will have a temporal autocorrelation, and if it equals "white.noise", it will be a pseudo-random sequence of numbers generated with \code{\link{rnorm}}, with no temporal autocorrelation. The importance of the random sequence in predicting the response is stored for each model run, and used as a benchmark to assess the importance of the other predictors.

Importance values of other predictors that are above the median of the importance of the random term should be interpreted as non-random, and therefore, significant.
}
\examples{
\donttest{
#loading data
data(palaeodataLagged)

# Simplified call - response and drivers auto-detected from attributes
memory.output <- computeMemory(
 lagged.data = palaeodataLagged,
 random.mode = "autocorrelated",
 repetitions = 10
)

str(memory.output)
str(memory.output$memory)

#plotting output
plotMemory(memory.output = memory.output)
}
}
\references{
\itemize{
  \item Wright, M. N. & Ziegler, A. (2017). ranger: A fast implementation of random forests for high dimensional data in C++ and R. J Stat Softw 77:1-17. \doi{10.18637/jss.v077.i01}.
  \item Breiman, L. (2001). Random forests. Mach Learn, 45:5-32. \doi{10.1023/A:1010933404324}.
  \item Hastie, T., Tibshirani, R., Friedman, J. (2009). The Elements of Statistical Learning. Springer, New York. 2nd edition.
  }
}
\seealso{
\code{\link{plotMemory}}, \code{\link{extractMemoryFeatures}}

Other memoria: 
\code{\link{extractMemoryFeatures}()},
\code{\link{plotMemory}()}
}
\author{
Blas M. Benito  <blasbenito@gmail.com>
}
\concept{memoria}
