#' Forecasting from NARFIMA-class Models
#'
#' This function produces forecasts from fitted models of class \code{narfima}, including NARFIMA, NARIMA, NBSTS, and NNaive. Multi-step forecasts are computed recursively. Prediction intervals can also be obtained through simulation.
#'
#' @param object An object of class "narfima".
#' @param h Number of periods to forecast. If \code{xreg} is provided, \code{h} is ignored and the forecast horizon is determined by the number of rows in \code{xreg}.
#' @param PI Logical value indicating whether to compute prediction intervals (default is \code{FALSE}).
#' @param level Confidence level for prediction intervals (default is \code{80}), ignored when \code{PI = FALSE}.
#' @param fan If \code{TRUE} (default is \code{FALSE}), level is set to \code{seq(51, 99, by = 3)}. This is suitable for fan plots.
#' @param bootstrap Logical value indicating whether to use bootstrapping for prediction intervals (default is \code{FALSE}), ignored when \code{PI = FALSE}.
#' @param npaths Number of simulation paths for prediction intervals (default is \code{1000}), ignored when \code{PI = FALSE}.
#' @param innov An optional vector of innovations to use for simulating future values (default is \code{NULL}). If \code{NULL} and \code{PI = TRUE}, random innovations are generated.
#' @param xreg An optional matrix or data frame of exogenous variables to be used in forecasting (default is \code{NULL}).
#' @param lambda Numeric value for the Box-Cox transformation parameter of \code{y} (default is taken from the fitted model, i.e., \code{object$lambda}).
#' @param lambdae Numeric value for the Box-Cox transformation parameter of \code{er} (default is taken from the fitted model, i.e., \code{object$lambdae}).
#' @param ... Additional arguments passed to \code{pred_interval_narfima}.
#'
#' @return An object of class "forecast" containing:
#' \item{mean}{The point forecasts.}
#' \item{lower}{The lower bounds of the prediction intervals.}
#' \item{upper}{The upper bounds of the prediction intervals.}
#' \item{level}{The confidence level for the prediction intervals.}
#' \item{model}{The fitted NARFIMA model.}
#' \item{series}{The name of the input series.}
#' \item{method}{A string describing the model parameters.}
#' \item{y}{The input time series.}
#' \item{e}{The transformed residual series}
#'
#' @importFrom forecast BoxCox InvBoxCox
#' @importFrom stats ar as.ts complete.cases frequency fitted is.ts na.omit predict quantile residuals rnorm runif sd ts tsp "tsp<-"
#' @importFrom utils tail
#'
#' @examples
#' h <- 3
#'
# # Take the last 83 observations from EuStockMarkets (CAC index): 80 points for training and the final 3 points for testing.
#' data <- EuStockMarkets[(nrow(EuStockMarkets) - 83):nrow(EuStockMarkets), 3]
#'
#' train <- data[1:(length(data) - h)]
#' test <- data[(length(data) - h + 1):length(data)]
#'
#' narfima_model <- auto_narfima(train)
#' narfima_forecast <- forecast_narfima_class(narfima_model, h)
#'
#' @export

forecast_narfima_class <- function(object, h = ifelse(object$m > 1, 2 * object$m, 10), PI = FALSE, level = 80, fan = FALSE, bootstrap = FALSE, npaths = 1000, innov = NULL, xreg = NULL, lambda = object$lambda, lambdae = object$lambdae, ...) {

  out <- object
  tspx <- tsp(out$y)

  if (!is.null(xreg) && !is.null(out$xreg)) {
    selected_columns <- intersect(colnames(xreg), colnames(out$xreg))
    xreg <- xreg[, selected_columns, drop = FALSE]
  }


  if (fan) {
    level <- seq(51, 99, by = 3)
  } else {
    if (min(level) > 0 && max(level) < 1) {
      level <- 100 * level
    } else if (min(level) < 0 || max(level) > 99.99) {
      stop("Confidence limit out of range")
    }
  }


  # Check if xreg was used in fitted model
  if (is.null(object$xreg)) {
    if (!is.null(xreg)) {
      warning("Exogenous variables were not used in fitted model, xreg will be ignored")
    }
    xreg <- NULL
  }

  else {
    if (is.null(xreg)) {
      stop("No exogenous variables provided")
    }

    xreg <- as.matrix(xreg)

    if (NCOL(xreg) != NCOL(object$xreg)) {
      stop("Number of exogenous variables does not match fitted model")
    }

    if(!identical(colnames(xreg), colnames(object$xreg))){
      warning("xreg contains different column names from the xreg used in training. Please check that the regressors are in the same order.")
    }

    h <- NROW(xreg)
  }


  fcast <- numeric(h)
  xx <- object$y
  ee <- object$e
  xxreg <- xreg


  if (!is.null(lambda)) {
    xx <- BoxCox(xx, lambda)
    lambda <- attr(xx, "lambda")
  }


  if (!is.null(lambdae)) {
    ee <- BoxCox(ee, lambdae)
    lambdae <- attr(ee, "lambdae")
  }


  # Check and apply scaling of fitted model
  if (!is.null(object$scaley)) {
    xx <- scale(xx, center = object$scaley$center, scale = object$scaley$scale)
    if (!is.null(xreg)) {
      xxreg <- scale(xreg, center = object$scalexreg$center, scale = object$scalexreg$scale)
    }
  }


  if (!is.null(object$scalee)) {
    ee <- scale(ee, center = object$scalee$center, scale = object$scalee$scale)
    if (!is.null(xreg)) {
      xxreg <- scale(xreg, center = object$scalexreg$center, scale = object$scalexreg$scale)
    }
  }


  # Get lags used in fitted model
  lags <- object$lags
  lagse <- object$lagse
  maxlag <- max(lags)
  maxlage <- max(lagse)
  flag <- rev(tail(xx, n = maxlag))
  flage <- rev(tail(ee, n = maxlage))


  # Iterative 1-step forecast
  for (i in 1:h){
    newdata <- c(flag[lags], flage[lagse], xxreg[i, ])
    if (any(is.na(newdata))) {
      stop("I can't forecast when there are missing values near the end of the series.")
    }
    fcast[i] <- mean(sapply(object$model, predict, newdata = newdata))
    flag <- c(fcast[i], flag[-maxlag])
    flage<- c(0, flage[-maxlage])
  }


  # Re-scale point forecasts
  if (!is.null(object$scaley)) {
    fcast <- fcast * object$scaley$scale + object$scaley$center
  }


  # Add ts properties
  fcast <- ts(fcast, start = tspx[2] + 1 / tspx[3], frequency = tspx[3])


  # Back-transform point forecasts
  if (!is.null(lambda)) {
    fcast <- InvBoxCox(fcast, lambda)
  }


  # Compute prediction intervals using simulations
  if (isTRUE(PI)) {
    nint <- length(level)
    sim <- matrix(NA, nrow = npaths, ncol = h)

    if (!is.null(innov)) {
      if (length(innov) != h * npaths) {
        stop("Incorrect number of innovations, need h*npaths values")
      }
      innov <- matrix(innov, nrow = h, ncol = npaths)
      bootstrap <- FALSE
    }

    for (i in 1:npaths)
      sim[i, ] <- pred_interval_narfima(object, nsim = h, bootstrap = bootstrap, xreg = xreg, lambda = lambda, innov = innov[, i], ...)
    lower <- apply(sim, 2, quantile, 0.5 - level / 200, type = 8, na.rm = TRUE)
    upper <- apply(sim, 2, quantile, 0.5 + level / 200, type = 8, na.rm = TRUE)

    if (nint > 1L) {
      lower <- ts(t(lower))
      upper <- ts(t(upper))
    }

    else {
      lower <- ts(matrix(lower, ncol = 1L))
      upper <- ts(matrix(upper, ncol = 1L))
    }
    out$lower <- future_msts(out$y, lower)
    out$upper <- future_msts(out$y, upper)
  }


  else {
    level <- NULL
    lower <- NULL
    upper <- NULL
  }

  out$mean <- future_msts(out$y, fcast)
  out$level <- level

  return(structure(out, class = "forecast"))
}


pred_interval_narfima <- function(object, nsim = length(object$y), future = TRUE, bootstrap = FALSE, innov = NULL, xreg = NULL, lambda = object$lambda, lambdae = object$lambdae, seed = NULL, ...) {

  if (is.null(innov)) {
    if (!is.null(seed)) {
      withr::local_seed(seed)
    }
  } else {
    nsim <- length(innov)
  }

  if (is.null(object$y)) {
    future <- FALSE
  }

  ## only future currently implemented
  if (!future) {
    warning("pred_interval_narfima() currently only supports future = TRUE")
  }


  ## set simulation innovations
  if (bootstrap) {
    res <- na.omit(c(residuals(object, type = "innovation")))
    res <- res - mean(res)

    ## scale if appropriate
    if (!is.null(object$scaley$scale)) {
      res <- res / object$scaley$scale
    }
    e <- sample(res, nsim, replace = TRUE)
  } else if (is.null(innov)) {
    res <- na.omit(c(residuals(object, type = "innovation")))

    ## scale if appropriate
    if (!is.null(object$scaley$scale)) {
      res <- res / object$scaley$scale
    }
    e <- rnorm(nsim, 0, sd(res, na.rm = TRUE))
  } else if (length(innov) == nsim) {
    e <- innov
    if (!is.null(object$scaley$scale)) {
      e <- e / object$scaley$scale
    }
  } else if (isTRUE(innov == 0L)) {
    ## to pass innov=0 so simulation equals mean forecast
    e <- rep(innov, nsim)
  } else {
    stop("Length of innov must be equal to nsim")
  }

  tspx <- tsp(object$y)


  # Check if xreg was used in fitted model
  if (is.null(object$xreg)) {
    if (!is.null(xreg)) {
      warning("Exogenous variables were not used in fitted model, xreg will be ignored")
    }
    xreg <- NULL
  } else {
    if (is.null(xreg)) {
      stop("No exogenous variables provided")
    }
    xreg <- as.matrix(xreg)
    if (NCOL(xreg) != NCOL(object$xreg)) {
      stop("Number of exogenous variables does not match fitted model")
    }
    if (NROW(xreg) != nsim) {
      stop("Number of rows in xreg does not match nsim")
    }
  }


  xx <- object$y
  if (!is.null(lambda)) {
    xx <- BoxCox(xx, lambda)
    lambda <- attr(xx, "lambda")
  }


  ee <- object$e
  if (!is.null(lambdae)) {
    ee <- BoxCox(ee, lambdae)
    lambdae <- attr(ee, "lambdae")
  }


  # Check and apply scaling of fitted model
  if (!is.null(object$scaley)) {
    xx <- scale(xx, center = object$scaley$center, scale = object$scaley$scale)
    if (!is.null(xreg)) {
      xreg <- scale(xreg, center = object$scalexreg$center, scale = object$scalexreg$scale)
    }
  }


  if (!is.null(object$scalee)) {
    ee <- scale(ee, center = object$scalee$center, scale = object$scalee$scale)
  }


  # Get lags used in fitted model
  lags <- object$lags
  maxlag <- max(lags)
  flag <- rev(tail(xx, n = maxlag))

  lagse <- object$lagse
  maxlagse <- max(lagse)
  flage <- rev(tail(ee, n = maxlagse))


  ## Simulate by iteratively forecasting and adding innovation
  path <- numeric(nsim)

  for (i in 1:nsim) {
    newdata <- c(flag[lags],flage[lagse], xreg[i, ])
    if (any(is.na(newdata))) {
      stop("I can't simulate when there are missing values near the end of the series.")
    }
    path[i] <- mean(sapply(object$model, predict, newdata = newdata)) + e[i]
    flag <- c(path[i], flag[-maxlag])
    flage <- c(path[i], flage)
  }


  ## Re-scale simulated points
  if (!is.null(object$scaley)) {
    path <- path * object$scaley$scale + object$scaley$center
  }


  ## Add ts properties
  path <- ts(path, start = tspx[2] + 1 / tspx[3], frequency = tspx[3])


  ## Back-transform simulated points
  if (!is.null(lambda)) {
    path <- InvBoxCox(path, lambda)
  }
  return(path)
}




future_msts <- function(x, y) {
  if (NCOL(y) > 1) {
    class(y) <- c("mts", "ts", "matrix")
  } else {
    class(y) <- "ts"
  }

  if ("msts" %in% class(x)) {
    class(y) <- c("msts", class(y))
  }

  attr <- attributes(x)
  attr$tsp[1:2] <- attr$tsp[2] + c(1, NROW(y)) / attr$tsp[3]
  attributes(y)$tsp <- attr$tsp
  attributes(y)$msts <- attr$msts
  return(y)
}
