#' Generate example data with five covariates
#'
#' This function creates some example data using the data generation process described
#' in simulation 1 of \insertCite{frazier2024bias}{nbpInference}. The dataframe
#' contains a treatment variable Z, outcome variable Y, and five covariates X1,...,X5.
#'
#' @param N Number of observations to simulate, which should be a positive
#'  whole number.
#' @return an N x 7 matrix containing treatment, outcome, and covariates.
#' @examples
#' generate.data.dose(N = 100)
#' @family data generation
#' @importFrom stats rnorm
#' @importFrom Rdpack reprompt
#' @export
generate.data.dose <- function(N){

  #### error messaging

  if(is.numeric(N) == FALSE){
    stop("N must be numeric.")
  }

  N <- round(N)

  if(N <= 0){
    stop("Sample size must be greater than 0.")
  }

  #### functionality

  x1 = rnorm(N, 0, 1); x2 = rnorm(N, 0, 1); x3 = rnorm(N, 0, 1); x4 = rnorm(N, 0, 1); x5 = rnorm(N, 0, 1);
  error1 = rnorm(N, 0, 1)
  z = x1 + x2^2 + abs(x3 * x4) + ifelse(x4 > 0, 1, 0) + log(1 + abs(x5)) + error1
  error2 = rnorm(N, 0, 3)
  y = z + 0.3*x1*z + 0.2*x3^3*z + exp(abs(x2 - x4)) - sin(x5) + error2
  data = data.frame("Z" = z, "Y" = y, "X1" = x1, "X2" = x2,
                    "X3" = x3, "X4" = x4, "X5" = x5)

  return(data)
}

#' Generate sample data with six covariates
#'
#' This function creates some example data using the data generation process
#' for the secondary set of simulations described in the supplementary materials
#' of \insertRef{frazier2024bias}{nbpInference}. The dataframe contains a treatment
#' variable Z, outcome variable Y, and five covariate X1,...,X6
#'
#' @param N Number of observations to simulate, which should be a positive
#'  whole number.
#' @return an N x 8 matrix containing treatment, outcome, and covariates.
#' @examples
#' generate.data.dose2(N = 100)
#' @family data generation
#' @importFrom stats rnorm rbinom rpois runif
#' @export
generate.data.dose2 <- function(N){

  #### error messaging

  if(is.numeric(N) == FALSE){
    stop("N must be numeric.")
  }

  N <- round(N)

  if(N <= 0){
    stop("Sample size must be greater than 0.")
  }

  #### functionality

  x1 = rbinom(N, 1, 0.5); x2 = rbinom(N, 10, 0.75); x3 = rpois(N, 1.5); x4 = runif(N, 0, 3); x5 = runif(N, -1, 1); x6 = runif(N, -5, 5);
  error1 = rnorm(N)
  z = ifelse(x1 == 1 & x2 > 5, 4, 0) - x1 + x3 + (sin(x5))^2 + 2*log(1 + x4) + 2*exp(-abs(x6)) + error1
  error2 = rnorm(N, 0, 3)
  y = z + 0.7*x6*z + 2*x5^3*z + ifelse(x1 == 1, x3*(2*x4 + 1), x3) + 0.5*x4*abs(x5) + x2 + error2
  data = data.frame("Z" = z, "Y" = y, "X1" = x1, "X2" = x2,
                    "X3" = x3, "X4" = x4, "X5" = x5, "X6" = x6)

  return(data)
}
