# This file contains the two-way examples from the book:
# Ishi-kuntz, M. Ordinal log-linear models. Thousand Oaks, CA: Sage.
# It shows how (sometimes mode than one way) the examples can be
# fit with the tools in the package.

emotional <- function() {
  emotional_support <- matrix(c( 785, 2119,
                                 1416, 2053),
                              nrow=2, byrow=TRUE)
  expected_N <- 6373
  expect_equal(sum(emotional_support), expected_N)
  expectedRowSums <- c(2904, 3469)
  expect_equal(rowSums(emotional_support), expectedRowSums)

  # Referred to as independence model in the book
  x <- log_linear_main_effect_design(emotional_support)
  result <- log_linear_fit(emotional_support, x)
  expect_true(abs(134.35 - result$g_squared) <= 0.005)
}


gender <- function() {
  gender_ideology <- matrix(c(466, 317, 203,
                               135, 198, 183,
                               51, 120, 160),
                             nrow=3, byrow=TRUE)

  expected_N <- 1833
  expect_equal(sum(gender_ideology), expected_N)
  expectedRowSums <- c(986, 516, 331)
  expect_equal(rowSums(gender_ideology), expectedRowSums)
  expectedColSums <- c(652, 635, 546)
  expect_equal(colSums(gender_ideology), expectedColSums)

  x <- log_linear_main_effect_design(gender_by_race)
  result <- log_linear_fit(gender_ideology, x)
  # book gives 153.22
  expect_true(abs(169.48 - result$g_squared) <= 0.005,
              info=paste(169.48, result$g_squared))

  resultb <- Goodman_null_association(gender_ideology)
  expect_true(abs(169.48 - resultb$g_squared) <= 0.005)

  # Referred to as a uniform association model in the book
  x <- log_linear_main_effect_design(gender_ideology)
  linear <- log_linear_create_linear_by_linear(gender_ideology)
  x_prime <- log_linear_append_column(x, linear)
  result2 <- log_linear_fit(gender_ideology, x_prime)
  # book gives 6.81
  expect_true(abs(6.59 - result2$g_squared) <= 0.005,
              info=paste(6.59, result2$g_squared))

  # Referred to as the RC model in the book
  result3 <- Goodman_model_ii(gender_ideology)
  # book gives 0.51
  expect_true(abs(0.22 - result3$g_squared) <= 0.005,
              info=paste(0.22, result3$g_squared), result3$chisq)
}

gender_race <- function() {
  gender_by_race <- matrix(c( 73,  67,  77,
                             464, 510, 443,
                             106,  47,  23),
                           nrow=3, byrow=TRUE)
  expected_N <- 1810
  expect_equal(sum(gender_by_race), expected_N)
  expectedRowSums <- c(217, 1417, 176)
  expect_equal(rowSums(gender_by_race), expectedRowSums)
  expectedColSums <- c(643, 624, 543)
  expect_equal(colSums(gender_by_race), expectedColSums)

  x <- log_linear_main_effect_design(gender_by_race)
  result <- log_linear_fit(gender_by_race, x)
  expect_true(abs(57.64 - result$g_squared) <= 0.005)

  resultb <- Goodman_null_association(gender_by_race)
  expect_true(abs(57.64 - resultb$g_squared) <= 0.005)

  # Referred to as rows effects model in the book
  result2 <- Goodman_model_i(gender_by_race, column_effects =FALSE)
  expect_true(abs(2.70 - result2$g_squared) <= 0.005)
}

run_all_examples <- function() {
  emotional()
  gender()
  gender_race()
}
