% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ollama_live.R
\name{ollama_compare_pair_live}
\alias{ollama_compare_pair_live}
\title{Live Ollama comparison for a single pair of samples}
\usage{
ollama_compare_pair_live(
  ID1,
  text1,
  ID2,
  text2,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  host = getOption("pairwiseLLM.ollama_host", "http://127.0.0.1:11434"),
  tag_prefix = "<BETTER_SAMPLE>",
  tag_suffix = "</BETTER_SAMPLE>",
  think = FALSE,
  num_ctx = 8192L,
  include_raw = FALSE,
  ...
)
}
\arguments{
\item{ID1}{Character ID for the first sample.}

\item{text1}{Character string containing the first sample's text.}

\item{ID2}{Character ID for the second sample.}

\item{text2}{Character string containing the second sample's text.}

\item{model}{Ollama model name (for example \code{"mistral-small3.2:24b"},
\code{"qwen3:32b"}, \code{"gemma3:27b"}).}

\item{trait_name}{Short label for the trait (for example
\code{"Overall Quality"}).}

\item{trait_description}{Full-text definition of the trait.}

\item{prompt_template}{Prompt template string, typically from
\code{\link[=set_prompt_template]{set_prompt_template()}}.}

\item{host}{Base URL of the Ollama server. Defaults to the option
\code{getOption("pairwiseLLM.ollama_host", "http://127.0.0.1:11434")}.}

\item{tag_prefix}{Prefix for the better-sample tag. Defaults to
\code{"<BETTER_SAMPLE>"}.}

\item{tag_suffix}{Suffix for the better-sample tag. Defaults to
\code{"</BETTER_SAMPLE>"}.}

\item{think}{Logical; if \code{TRUE} and the model is a Qwen model (name
starts with \code{"qwen"}), the temperature is set to \code{0.6}.
Otherwise the temperature is \code{0}. The \code{think} argument does
not itself modify the HTTP request body; it is used only for choosing
the temperature, but the function will parse a \code{thinking} field
from the response whenever one is present.}

\item{num_ctx}{Integer; context window to use via \code{options$num_ctx}.
The default is \code{8192L}.}

\item{include_raw}{Logical; if \code{TRUE}, adds a list-column
\code{raw_response} containing the parsed JSON body returned by Ollama
(or \code{NULL} on parse failure). This is useful for debugging.}

\item{...}{Reserved for future extensions.}
}
\value{
A tibble with one row and columns:

\itemize{
\item \code{custom_id} – ID string of the form
\code{"LIVE_<ID1>_vs_<ID2>"}.
\item \code{ID1}, \code{ID2} – the sample IDs supplied to the function.
\item \code{model} – model name reported by the API (or the requested
model).
\item \code{object_type} – backend object type (for example
\code{"ollama.generate"}).
\item \code{status_code} – HTTP-style status code (\code{200} if
successful).
\item \code{error_message} – error message if something goes wrong;
otherwise \code{NA}.
\item \code{thoughts} – reasoning / thinking text when a
\code{thinking} field is returned by Ollama; otherwise \code{NA}.
\item \code{content} – visible response text from the model (from the
\code{response} field).
\item \code{better_sample} – \code{"SAMPLE_1"}, \code{"SAMPLE_2"}, or
\code{NA}, based on tags found in \code{content}.
\item \code{better_id} – \code{ID1} if \code{"SAMPLE_1"} is chosen,
\code{ID2} if \code{"SAMPLE_2"} is chosen, otherwise \code{NA}.
\item \code{prompt_tokens} – prompt / input token count (if reported).
\item \code{completion_tokens} – completion / output token count (if
reported).
\item \code{total_tokens} – total token count (if reported).
\item \code{raw_response} – optional list-column containing the parsed
JSON body (present only when \code{include_raw = TRUE}).
}
}
\description{
\code{ollama_compare_pair_live()} sends a single pairwise comparison prompt to a
local Ollama server and parses the result into the standard pairwiseLLM
tibble format.
}
\details{
The function targets the \verb{/api/generate} endpoint on a running Ollama
instance and expects a single non-streaming response. Model names should
match those available in your Ollama installation (for example
\code{"mistral-small3.2:24b"}, \code{"qwen3:32b"}, \code{"gemma3:27b"}).

Temperature and context length are controlled as follows:

\itemize{
\item By default, \code{temperature = 0} for all models.
\item For Qwen models (model names beginning with \code{"qwen"}) and
\code{think = TRUE}, \code{temperature} is set to \code{0.6}.
\item The context window is set via \code{options$num_ctx}, which
defaults to \code{8192L} but may be overridden via the \code{num_ctx}
argument.
}

If the Ollama response includes a \code{thinking} field (as described in the
Ollama API), that string is stored in the \code{thoughts} column of the
returned tibble; otherwise \code{thoughts} is \code{NA}. This allows
pairwiseLLM to consume Ollama's native thinking output in a way that is
consistent with other backends that expose explicit reasoning traces.

The Ollama backend is intended to be compatible with the existing OpenAI,
Anthropic, and Gemini backends, so the returned tibble can be used
directly with downstream helpers such as \code{\link[=build_bt_data]{build_bt_data()}} and
\code{\link[=fit_bt_model]{fit_bt_model()}}.

In typical workflows, users will call \code{\link[=llm_compare_pair]{llm_compare_pair()}} with
\code{backend = "ollama"} rather than using
\code{ollama_compare_pair_live()} directly. The direct helper is exported
so that advanced users can work with Ollama in a more explicit and
backend-specific way.

The function assumes that:

\itemize{
\item An Ollama server is running and reachable at \code{host}.
\item The requested \code{model} has already been pulled, for example
via \code{ollama pull mistral-small3.2:24b} on the command line.
}

When the Ollama response includes a \code{thinking} field (as documented
in the Ollama API), that string is copied into the \code{thoughts} column
of the returned tibble; otherwise \code{thoughts} is \code{NA}. This
parsed thinking output can be logged, inspected, or analyzed alongside
the visible comparison decisions.
}
\examples{
\dontrun{
# Requires a running Ollama server and locally available models.

data("example_writing_samples", package = "pairwiseLLM")

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

ID1 <- example_writing_samples$ID[1]
ID2 <- example_writing_samples$ID[2]
text1 <- example_writing_samples$text[1]
text2 <- example_writing_samples$text[2]

# Make sure an Ollama server is running

# mistral example
res_mistral <- ollama_compare_pair_live(
  ID1               = ID1,
  text1             = text1,
  ID2               = ID2,
  text2             = text2,
  model             = "mistral-small3.2:24b",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl
)

res_mistral$better_id

# qwen example with reasoning
res_qwen_think <- ollama_compare_pair_live(
  ID1               = ID1,
  text1             = text1,
  ID2               = ID2,
  text2             = text2,
  model             = "qwen3:32b",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  think             = TRUE,
  include_raw       = TRUE
)

res_qwen_think$better_id
res_qwen_think$thoughts
}

}
\seealso{
\itemize{
\item \code{\link[=submit_ollama_pairs_live]{submit_ollama_pairs_live()}} for single-backend, row-wise comparisons.
\item \code{\link[=llm_compare_pair]{llm_compare_pair()}} for backend-agnostic single-pair comparisons.
\item \code{\link[=submit_llm_pairs]{submit_llm_pairs()}} for backend-agnostic comparisons over tibbles of
pairs.
}
}
