#' Calculate Observed Proportion of Concordant Pairs (OPCP)
#'
#' This function provides a matrix that contains Pearson correlations in the lower triangle and Observed Proportions of Concordant Pairs (OPCPs) in the upper triangle between multiple variables.
#'
#' @param data A data frame containing the variables specified in the formula.
#' @param round_digits The number of decimals to be included in the resulting table.
#'
#' @return A matrix of correlations and OPCPs.
#' @examples
#' # Example using the spi dataset from the psychTools package
#' sc <- psych::scoreVeryFast(psychTools::spi.keys, psychTools::spi)
#' spi_sc <- cbind(psychTools::spi, sc)
#' spi_sc_age_sex_B5 <- spi_sc |>
#'   dplyr::select(age, sex, Agree, Consc, Neuro, Extra, Open)
#'
#'
#' OPCP_mat(data = spi_sc_age_sex_B5)
#'
#' @importFrom stats cor sd
#'
#' @export
OPCP_mat <- function(data, round_digits = 2) {
  # Compute Kendall's tau and transform to OPCP
  tau_matrix <- cor(data, method = "kendall", use = "pairwise.complete.obs")
  opcp_matrix <- ((tau_matrix / 2) + 0.5) * 100  # OPCP * 100

  # Compute Pearson correlation matrix
  pearson_matrix <- cor(data, method = "pearson", use = "pairwise.complete.obs")

  # Get means and standard deviations
  means <- round(colMeans(data, na.rm = TRUE), round_digits)
  sds <- round(apply(data, 2, sd, na.rm = TRUE), round_digits)

  # Setup result matrix
  vars <- colnames(data)
  n <- length(vars)
  result_matrix <- matrix("", nrow = n, ncol = n + 2,
                          dimnames = list(vars, c("Mean", "SD", vars)))

  # Fill in means and SDs
  result_matrix[, "Mean"] <- format(means, nsmall = round_digits)
  result_matrix[, "SD"] <- format(sds, nsmall = round_digits)

  # Fill in diagonal with 1.00
  for (i in seq_len(n)) {
    result_matrix[i, i + 2] <- format(1, nsmall = round_digits)
  }

  # Fill upper triangle with OPCP
  for (i in 1:(n - 1)) {
    for (j in (i + 1):n) {
      result_matrix[i, j + 2] <- format(round(opcp_matrix[i, j], round_digits), nsmall = round_digits)
    }
  }

  # Fill lower triangle with Pearson
  for (i in 2:n) {
    for (j in 1:(i - 1)) {
      result_matrix[i, j + 2] <- format(round(pearson_matrix[i, j], round_digits), nsmall = round_digits)
    }
  }

  result_matrix <- as.data.frame(result_matrix)

  return(result_matrix)
}



