#' @title Higuchi Drug Release Kinetic Model
#' @name higuchi_release
#' @description
#' Fits experimental cumulative drug release data to the Higuchi square-root
#' kinetic model using linear regression of cumulative percent drug released versus
#' square root of time. The function supports optional grouping variables
#' (e.g., formulation, batch) and optional pH-dependent analysis. It generates
#' publication-quality plots with experimental curves, fitted Higuchi straight
#' lines, and annotations for Higuchi release constant (kH), intercept,
#' coefficient of determination (R^2), and the time required for 50-percent drug
#' release (t50).
#'
#' @param data A data frame containing experimental drug release data.
#' @param sqrt_time_col Character string specifying the column name for square
#' root of time.
#' @param release_col Character string specifying the column name for
#' cumulative percent drug released.
#' @param group_col Optional character string specifying a grouping variable
#' (e.g., formulation, batch). Default is NULL.
#' @param pH_col Optional character string specifying a column containing pH
#' values. If provided, Higuchi models are fitted separately for each pH.
#' @param plot Logical; if TRUE, generates a plot (default = TRUE).
#' @param annotate Logical; if TRUE, annotates the plot with kH, intercept, R^2,
#' and t50 values (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text geom_smooth labs
#' theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{A data frame with kH, intercept, R^2, and t50
#'         values for each group or pH condition.}
#'   \item{\code{data}}{The processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single Formulation
#' df <- data.frame(
#'   sqrt_time = c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
#'   release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
#' )
#' higuchi_release(
#'   data = df,
#'   sqrt_time_col = "sqrt_time",
#'   release_col = "release"
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df_2 <- data.frame(
#'   sqrt_time = c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
#'   release = c(
#'     0, 11.4, 20.8, 30.8, 39.8, 57.8, 72.0, 84.8, 93.5,   # Formulation A
#'     0, 10.2, 18.6, 29.7, 37.8, 56.5, 71.9, 83.7, 92.9   # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 9)
#' )
#' higuchi_release(
#'   data = df_2,
#'   sqrt_time_col = "sqrt_time",
#'   release_col = "release",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent Higuchi release
#' df_pH <- data.frame(
#'   sqrt_time = rep(
#'     c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
#'     2
#'   ),
#'   release = c(
#'     0, 11.4, 20.8, 30.8, 39.8, 57.8, 72.0, 84.8, 93.5,  # pH 7.4
#'     0, 17.2, 23.8, 35.5, 41.5, 58.3, 73.6, 86.2, 93.1   # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 9)
#' )
#' higuchi_release(
#'   data = df_pH,
#'   sqrt_time_col = "sqrt_time",
#'   release_col = "release",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   sqrt_time = rep(c(0.0, 2.5, 4.0, 5.2, 6.3, 7.4, 8.6, 9.8, 11.0, 12.2), 2),
#'   release = c(
#'     0.0, 12.5, 21.8, 31.2, 39.6, 50.8, 63.5, 74.2, 84.9, 92.8,  # pH 7.4
#'     0.0, 9.8, 18.6, 26.9, 34.7, 45.3, 56.8, 67.9, 77.4, 85.2   # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 10)
#' )
#' df2 <- data.frame(
#'   sqrt_time = rep(c(0.0, 2.5, 4.0, 5.2, 6.3, 7.4, 8.6, 9.8, 11.0, 12.2), 2),
#'   release = c(
#'     0.0, 11.3, 20.4, 29.1, 37.8, 48.6, 60.1, 71.0, 81.5, 89.6,  # pH 7.4
#'     0.0, 8.9, 16.7, 24.6, 32.1, 42.5, 53.4, 64.0, 73.1, 80.8   # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 10)
#' )
#' df_all <- rbind(
#'   cbind(formulation = "Dataset 1", df1),
#'   cbind(formulation = "Dataset 2", df2)
#' )
#' higuchi_release(
#'   data = df_all,
#'   sqrt_time_col = "sqrt_time",
#'   release_col = "release",
#'   group_col = "formulation",
#'   pH_col = "pH"
#' )
#' @references Higuchi, T. (1963) <doi:10.1002/jps.2600521210> Mechanism of
#' sustained-action medication. Journal of Pharmaceutical Sciences, 52(12),
#' 1145–1149.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("sqrt_time", "release", "group", "kH", "intercept", "R2",
                         "t50", "label", "x_pos", "y_pos", "hjust", "vjust"))

higuchi_release <- function(data,
                            sqrt_time_col = "sqrt_time",
                            release_col = "release",
                            group_col = NULL,
                            pH_col = NULL,
                            plot = TRUE,
                            annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required.")
  }

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(sqrt_time_col, release_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)

  colnames(df)[1:2] <- c("sqrt_time", "release")

  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }

  df$group <- as.factor(df$group)

  # -------------------------
  # Higuchi model fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::lm(release ~ sqrt_time, data = d)
    s <- summary(fit)

    kH <- coef(fit)[2]
    intercept <- coef(fit)[1]
    R2 <- s$r.squared

    t50 <- if (!is.na(kH) && kH > 0 && intercept < 50) {
      ((50 - intercept) / kH)^2
    } else {
      NA_real_
    }

    data.frame(
      group = unique(d$group),
      kH = kH,
      intercept = intercept,
      R2 = R2,
      t50 = t50
    )

  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(df, ggplot2::aes(x = sqrt_time, y = release, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1.2) +
      ggplot2::geom_smooth(
        method = "lm",
        se = FALSE,
        color = "black",
        linewidth = 1
      ) +
      ggplot2::labs(
        title = "Higuchi Drug Release Kinetics",
        subtitle = "Experimental data with Higuchi square-root model fit",
        x = expression(sqrt(Time)),
        y = "Cumulative % Drug Released",
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)

    if (annotate && num_groups <= 2) {

      ann <- fit_results
      ann$label <- paste0(
        "kH = ", round(ann$kH, 3), " %*min^-1/2\n",
        "Intercept = ", round(ann$intercept, 2), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t[50] = ", round(ann$t50, 1), " min"
      )

      x_min <- min(df$sqrt_time)
      x_max <- max(df$sqrt_time)
      y_min <- min(df$release)
      y_max <- max(df$release)

      ann$x_pos <- c(
        x_min + 0.05 * (x_max - x_min),
        x_max - 0.05 * (x_max - x_min)
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_max - 0.05 * (y_max - y_min),
        y_min + 0.05 * (y_max - y_min)
      )[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(
            x = x_pos,
            y = y_pos,
            label = label,
            color = group
          ),
          hjust = ann$hjust,
          vjust = ann$vjust,
          size = 4,
          show.legend = FALSE
        )

    } else if (annotate && num_groups > 2) {
      message("More than 2 groups detected - annotations disabled to prevent overlap.")
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
