#' @title Two-Compartment IV Bolus Pharmacokinetic Model (Nonlinear)
#' @name two_compartment_iv_bolus_nl
#' @description
#' Fits plasma concentration-time data to a two-compartment intravenous (IV) bolus
#' pharmacokinetic model. The model assumes instantaneous drug administration and
#' distribution between central and peripheral compartments, with first-order elimination.
#'
#' Model:
#' C(t) = A * exp(-alpha * t) + B * exp(-beta * t)
#'
#' where:
#' \itemize{
#'   \item A, B: intercept coefficients
#'   \item alpha: distribution rate constant (alpha > beta)
#'   \item beta: elimination rate constant
#' }
#'
#' @param data A data frame containing plasma concentration-time data.
#' @param time_col Character string specifying the column name for time.
#' @param conc_col Character string specifying the column name for plasma concentration.
#' @param dose Numeric value specifying the administered IV bolus dose.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, subject).
#' @param plot Logical; if TRUE, generates a concentration-time plot with fitted curves.
#' @param annotate Logical; if TRUE, annotates the plot with PK parameters (only if <=2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm nls na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text labs theme
#' theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with biexponential coefficients
#'   (A and B), alpha, beta, t_half_alpha, t_half_beta, and R^2 for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject two-compartment IV bolus data
#' df <- data.frame(
#'   time = c(0.08, 0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(40.0, 30.5, 25.0, 17.5, 10.2, 6.4, 4.1, 2.8, 1.5)
#' )
#' two_compartment_iv_bolus_nl(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent pharmacokinetics (e.g., physiological state)
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     25.3, 22.1, 18.5, 13.2, 8.5, 5.6, 3.8,   # Condition A
#'     20.7, 18.0, 14.9, 11.3, 7.1, 4.7, 3.2    # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 7)
#' )
#' two_compartment_iv_bolus_nl(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects (population-style two-compartment IV bolus pharmacokinetics)
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 5),
#'   concentration = c(
#'     26.1, 23.2, 19.6, 14.0, 9.0, 6.0, 4.0,   # Subject 1
#'     24.8, 21.8, 18.4, 13.3, 8.8, 5.8, 3.9,   # Subject 2
#'     25.5, 22.5, 19.0, 13.8, 8.7, 5.7, 3.7,   # Subject 3
#'     23.9, 20.9, 17.7, 12.8, 8.4, 5.5, 3.5,   # Subject 4
#'     24.4, 21.5, 18.0, 13.0, 8.5, 5.6, 3.6    # Subject 5
#'   ),
#'   subject = rep(paste0("S", 1:5), each = 7)
#' )
#' two_compartment_iv_bolus_nl(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded. Swedish
#' Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "group", "A", "B", "alpha", "beta",
                         "t_half_alpha", "t_half_beta", "R2", "label", "x_pos",
                         "y_pos", "hjust", "vjust"))

two_compartment_iv_bolus_nl <- function(data,
                                        time_col = "time",
                                        conc_col = "concentration",
                                        dose,
                                        group_col = NULL,
                                        plot = TRUE,
                                        annotate = TRUE) {
  if (!requireNamespace("ggplot2", quietly = TRUE)) stop("Package 'ggplot2' is required.")

  # Prepare data
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  # Filter positive times and concentrations
  df <- df[df$time >= 0 & df$conc > 0, ]

  # Grouping
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("Experimental")
  }

  # Helper: Two-compartment model function
  twocomp_model <- function(time, A, B, alpha, beta) {
    A * exp(-alpha * time) + B * exp(-beta * time)
  }

  # Model fitting per group using nonlinear least squares
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {
    # Provide starting values based on data heuristics
    # Rough initial estimates:
    C0 <- max(d$conc)
    # Use biexponential decay heuristic for rate constants
    # Use linear fit on log(conc) for early and late points for rough alpha and beta

    early_points <- d[d$time <= quantile(d$time, 0.3), ]
    late_points <- d[d$time >= quantile(d$time, 0.7), ]

    if (nrow(early_points) < 2) early_points <- d[1:2, ]
    if (nrow(late_points) < 2) late_points <- d[(nrow(d)-1):nrow(d), ]

    fit_early <- stats::lm(log(conc) ~ time, data = early_points)
    fit_late <- stats::lm(log(conc) ~ time, data = late_points)

    alpha_start <- -coef(fit_early)[2]
    beta_start <- -coef(fit_late)[2]

    if (beta_start > alpha_start) {
      temp <- alpha_start
      alpha_start <- beta_start
      beta_start <- temp
    }

    A_start <- C0 / 2
    B_start <- C0 / 2

    # Bounds for parameters
    lower_bounds <- c(A = 0, B = 0, alpha = 1e-4, beta = 1e-4)
    upper_bounds <- c(A = Inf, B = Inf, alpha = Inf, beta = Inf)

    # Try fitting with nls (port algorithm allows bounds)
    fit <- tryCatch({
      stats::nls(conc ~ A * exp(-alpha * time) + B * exp(-beta * time),
          data = d,
          start = list(A = A_start, B = B_start, alpha = alpha_start, beta = beta_start),
          algorithm = "port",
          lower = lower_bounds,
          upper = upper_bounds,
          control = nls.control(maxiter = 500))
    }, error = function(e) NULL)

    if (is.null(fit)) {
      warning(paste("Model fitting failed for group", unique(d$group)))
      return(data.frame(group = unique(d$group),
                        A = NA, B = NA, alpha = NA, beta = NA,
                        t_half_alpha = NA, t_half_beta = NA, R2 = NA))
    }

    coef_fit <- coef(fit)

    # Calculate half-lives
    t_half_alpha <- log(2) / coef_fit["alpha"]
    t_half_beta <- log(2) / coef_fit["beta"]

    # Calculate fitted values and R squared
    pred <- predict(fit)
    rss <- sum((d$conc - pred)^2)
    tss <- sum((d$conc - mean(d$conc))^2)
    r_squared <- 1 - rss/tss

    data.frame(
      group = unique(d$group),
      A = coef_fit["A"],
      B = coef_fit["B"],
      alpha = coef_fit["alpha"],
      beta = coef_fit["beta"],
      t_half_alpha = t_half_alpha,
      t_half_beta = t_half_beta,
      R2 = r_squared
    )
  }))

  # Plotting
  if (plot) {
    pred_df <- do.call(rbind, lapply(split(df, df$group), function(d) {
      pars <- fit_results[fit_results$group == unique(d$group), ]
      t_seq <- seq(min(d$time), max(d$time), length.out = 200)
      data.frame(
        time = t_seq,
        conc = twocomp_model(t_seq, pars$A, pars$B, pars$alpha, pars$beta),
        group = pars$group
      )
    }))

    p <- ggplot2::ggplot(df, ggplot2::aes(time, conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(data = pred_df, ggplot2::aes(time, conc, color = group), linewidth = 1) +
      ggplot2::labs(
        title = "Two-Compartment IV Bolus Pharmacokinetic Model",
        subtitle = "Biexponential decline",
        x = "Time (hours)",
        y = "Plasma Concentration (mg/L)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # Annotations
    if (annotate && nrow(fit_results) <= 1) {
      ann <- fit_results
      ann$label <- paste0(
        "A = ", signif(ann$A, 3), "\n",
        "B = ", signif(ann$B, 3), "\n",
        "alpha = ", signif(ann$alpha, 3), "\n",
        "beta = ", signif(ann$beta, 3), "\n",
        "t1/2 alpha = ", round(ann$t_half_alpha, 2), "\n",
        "t1/2 beta = ", round(ann$t_half_beta, 2), "\n",
        "R2 = ", round(ann$R2, 3)
      )

      # Position all annotations in the upper right corner, stacked vertically
      x_max <- max(df$time)
      y_max <- max(df$conc)

      ann$x_pos <- x_max - 0.05 * (x_max)       # slightly left from max time
      # Stack annotations downwards with some spacing
      ann$y_pos <- y_max * (1 - 0.1 * (seq_len(nrow(ann)) - 1))

      ann$hjust <- 1   # right align text
      ann$vjust <- 1   # align top of text at y_pos

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
