% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sigmoid_emax.R
\name{sigmoid_emax}
\alias{sigmoid_emax}
\title{Sigmoid Emax (Stimulatory Hill Model) for Dose-Response Analysis}
\arguments{
\item{data}{A data frame containing dose-response experimental data.}

\item{dose_col}{Character string specifying the column name for dose or concentration.}

\item{response_col}{Character string specifying the column name for measured response.}

\item{group_col}{Optional character string specifying a column for grouping.}

\item{log_dose}{Logical; if TRUE, dose values are log10-transformed for plotting
(model fitting uses original dose values).}

\item{plot}{Logical; if TRUE, generates a dose-response plot with fitted Hill curves.}

\item{annotate}{Logical; if TRUE, annotates the plot with model parameters and fit metrics
(only if <=1 group).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with E0, Emax, EC50, Hill coefficient (\code{n}),
        RMSE, AIC, and BIC values for each group.}
  \item{\code{data}}{The processed dataset used for model fitting and plotting.}
}
}
\description{
Fits pharmacodynamic dose-response data to the stimulatory Hill
(Sigmoid Emax) model using nonlinear least squares regression.

The stimulatory Hill model describes increasing pharmacological
effects with increasing dose (or concentration) according to:

\deqn{
  E = E_0 + \frac{E_{max} \cdot D^n}{EC_{50}^n + D^n}
}

where \eqn{E_0} is the baseline effect, \eqn{E_{max}} is the maximum
stimulatory effect, \eqn{EC_{50}} is the dose producing 50% of
\eqn{E_{max}}, and \eqn{n} is the Hill coefficient.

This model is appropriate when the observed response increases
monotonically with dose.
}
\examples{
# Example I: Single dose-response dataset
df <- data.frame(
  dose = c(0.1, 0.3, 1, 3, 10, 30, 100),
  response = c(5, 10, 25, 55, 80, 92, 98)
)
sigmoid_emax(
  data = df,
  dose_col = "dose",
  response_col = "response"
)

# Example II: Two treatment groups
df2 <- data.frame(
  dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 2),
  response = c(
    3, 8, 20, 45, 70, 85,     # Group A
    2, 6, 15, 35, 60, 78      # Group B
  ),
  treatment = rep(c("Group A", "Group B"), each = 6)
)
sigmoid_emax(
  data = df2,
  dose_col = "dose",
  response_col = "response",
  group_col = "treatment",
  log_dose = TRUE
)

# Example III: Multiple subjects (population-style dose-response pharmacodynamics)
df_subjects <- data.frame(
  dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 5),
  response = c(
    5, 13, 30, 56, 80, 92,   # Subject 1
    4, 12, 28, 54, 78, 90,   # Subject 2
    6, 15, 33, 59, 83, 95,   # Subject 3
    5, 14, 31, 57, 81, 93,   # Subject 4
    3, 11, 26, 52, 76, 88    # Subject 5
  ),
  subject = rep(paste0("S", 1:5), each = 6)
)
sigmoid_emax(
  data = df_subjects,
  dose_col = "dose",
  response_col = "response",
  group_col = "subject",
  log_dose = TRUE
)
}
\references{
Hill, A. V. (1910) The possible effects of the aggregation of the
molecules of hæmoglobin on its dissociation curves. The Journal of Physiology,
40(4), iv–vii.

Holford, N. H. G. & Sheiner, L. B. (1981) <doi:10.2165/00003088-198106060-00002>
Understanding the dose-effect relationship. Clinical Pharmacokinetics, 6(6), 429–453.
}
\author{
Paul Angelo C. Manlapaz
}
