#' @title
#' Building fences.
#'
#' @description
#' Builds a fence, from an unordered collection of elements.
#'
#' @param elements A list of character strings (the names of the fence elements).
#'
#' @param orientation Either "upFirst" (the first element dominates the second) or
#' "downFirst" (the first element is dominated by the second).
#'
#' @return
#' A fence, an object of S4 class `POSet`.
#'
#' @examples
#' elems <- c("a", "b", "c", "d", "e")
#' fence <- FencePOSet(elems, orientation="upFirst")
#'
#' @name FencePOSet
#' @export FencePOSet
FencePOSet <- function(elements, orientation="upFirst") {
  if (!is.character(elements)) {
    stop("elements must be character array")
  }
  if (length(unique(elements)) != length(elements)) {
    stop("elements contains duplicated values")
  }
  if ((!methods::is(orientation, "character") || !(orientation %in% c('upFirst', 'downFirst')))) {
    out_str <- paste("Arguments must be a string either upFirst or downFirst ")
    stop(out_str)
  }

  tryCatch({
    ptr <- .Call("_BuildFencePOSet", elements, as.integer(orientation == 'upFirst'))
    result <- methods::new("POSet", ptr = ptr)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
