% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meds.R
\name{medications_remaining}
\alias{medications_remaining}
\title{Calculate remaining medications to be taken}
\usage{
medications_remaining(
  meds = NULL,
  group = c("medication", "format"),
  on_date = Sys.Date(),
  until_date = NULL
)
}
\arguments{
\item{meds}{Data frame containing medication schedule. Must have the following columns:
\itemize{
\item medication (character/factor): Name of the medication
\item format (character/factor): Format of the medication (e.g., pill, injection)
\item quantity (numeric): Number of units to take per day
\item start_date (Date or character string representing a date, e.g. "YYYY-MM-DD"): Date to start taking the medication
\item stop_date (Date or character string representing a date, e.g. "YYYY-MM-DD"): Final date on which the medication is taken
}

If NULL, will try to use the "pregnancy.medications" option. Required if option not set.}

\item{group}{Character string specifying how to group the results. One of:
\itemize{
\item "medication": Group by medication name (default)
\item "format": Group by medication format
}}

\item{on_date}{Date or character string representing a date, e.g. "YYYY-MM-DD",
specifying the date from which to calculate remaining medications.
Defaults to current system date}

\item{until_date}{Date or character string representing a date, e.g. "YYYY-MM-DD",
specifying cut-off date for remaining medications.
If NULL, defaults to the latest \code{stop_date} in \code{medications}.}
}
\value{
Returns a data frame containing remaining quantities, grouped as specified.
Assumes that the function is being called first thing in the day,
i.e. before any of \code{on_date}'s medications have been taken.
The data frame has two columns:
\itemize{
\item Either 'medication' or 'format' depending on grouping
\item quantity: Total number of units remaining
}

Only medications with remaining quantities > 0 are included.

If no medications remain, a message is printed to the console indicating this,
and a data frame with 0 rows is returned invisibly.
}
\description{
Calculates and displays how many medications remain to be taken as of a specific date,
based on a schedule of medications with start and stop dates. Results can be grouped
either by medication name or by format (e.g., tablet, injection).
}
\details{
If \code{on_date}, \code{until_date} \code{start_date} or \code{stop_date} is a character vector, the conversion to a \code{Date}
is handled by \code{anytime::anydate()}.

If any \code{start_date} is \code{NA} in any row, that row will \strong{not} be counted in the remaining quantities.
If any \code{stop_date} is \code{NA}, it throws an error.
}
\section{Global Options}{

\itemize{
\item \code{pregnancy.medications}: Data frame setting default medication schedule
}
}

\examples{
# Define medications table
#' # Create example medication schedule
meds <- data.frame(
  medication = c("progynova", "prednisolone", "clexane"),
  format = c("tablet", "tablet", "injection"),
  quantity = c(3, 2, 1),
  start_date = c("2025-04-21", "2025-04-26", "2025-05-08"),
  stop_date = c("2025-04-30", "2025-05-07", "2025-09-05")
)

# Calculate remaining medications
medications_remaining(meds, on_date = "2025-04-21")

medications_remaining(meds, group = "format", on_date = "2025-04-21")

# Calculate medications for a specified period
medications_remaining(
  meds = meds,
  on_date = "2025-04-23",
  until_date = "2025-04-30"
)

# Set and use global medications option
#' Store original medications setting (without message)
original_medications <- getOption("pregnancy.medications")
set_medications(pregnancy::medications)
medications_remaining(on_date = as.Date("2025-05-01"))

# Restore original medications setting (without message)
options(pregnancy.medications = original_medications)

}
\seealso{
\itemize{
\item \code{\link[=set_medications]{set_medications()}} for setting default medication schedule
\item \code{\link[=get_medications]{get_medications()}} for retrieving current medication schedule
\item \link{medications} for an example medications data frame
}
}
