% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find_peptide_in_structure.R
\name{find_peptide_in_structure}
\alias{find_peptide_in_structure}
\title{Finds peptide positions in a PDB structure based on positional matching}
\usage{
find_peptide_in_structure(
  peptide_data,
  peptide,
  start,
  end,
  uniprot_id,
  pdb_data = NULL,
  retain_columns = NULL
)
}
\arguments{
\item{peptide_data}{a data frame containing at least the input columns to this function.}

\item{peptide}{a character column in the \code{peptide_data} data frame that contains the
sequence or any other unique identifier for the peptide that should be found.}

\item{start}{a numeric column in the \code{peptide_data} data frame that contains start positions
of peptides.}

\item{end}{a numeric column in the \code{peptide_data} data frame that contains end positions of
peptides.}

\item{uniprot_id}{a character column in the \code{peptide_data} data frame that contains UniProt
identifiers that correspond to the peptides.}

\item{pdb_data}{optional, a data frame containing data obtained with \code{fetch_pdb()}. If not
provided, information is fetched automatically. If this function should be run multiple times
it is faster to fetch the information once and provide it to the function. If provided, make
sure that the column names are identical to the ones that would be obtained by calling \code{fetch_pdb()}.}

\item{retain_columns}{a vector indicating if certain columns should be retained from the input
data frame. Default is not retaining additional columns \code{retain_columns = NULL}. Specific
columns can be retained by providing their names (not in quotations marks, just like other
column names, but in a vector).}
}
\value{
A data frame that contains peptide positions in the corresponding PDB structures. If a
peptide is not found in any structure or no structure is associated with the protein, the data
frame contains NAs values for the output columns. The data frame contains the following and
additional columns:
\itemize{
\item auth_asym_id: Chain identifier provided by the author of the structure in order to
match the identification used in the publication that describes the structure.
\item label_asym_id: Chain identifier following the standardised convention for mmCIF files.
\item peptide_seq_in_pdb: The sequence of the peptide mapped to the structure. If the
peptide only maps partially, then only the part of the sequence that maps on the structure is
returned.
\item fit_type: The fit type is either "partial" or "fully" and it indicates if the complete
peptide or only part of it was found in the structure.
\item start_adjusted: The adjusted start position of the peptide if the peptide was only partially
covered by the structure. If the peptide is fully covered it is just the provided start position.
\item end_adjusted: The adjusted end position of the peptide if the peptide was only partially
covered by the structure. If the peptide is fully covered it is just the provided end position.
\item label_seq_id_start: Contains the first residue position of the peptide in the structure
following the standardised convention for mmCIF files.
\item label_seq_id_end: Contains the last residue position of the peptide in the structure
following the standardised convention for mmCIF files.
\item auth_seq_id_start: Contains the first residue position of the peptide in the structure
based on the alternative residue identifier provided by the author of the structure in order
to match the identification used in the publication that describes the structure. This does
not need to be numeric and is therefore of type character.
\item auth_seq_id_end: Contains the last residue position of the peptide in the structure
based on the alternative residue identifier provided by the author of the structure in order
to match the identification used in the publication that describes the structure. This does
not need to be numeric and is therefore of type character.
\item auth_seq_id: Contains all positions (separated by ";") of the peptide in the structure
based on the alternative residue identifier provided by the author of the structure in order
to match the identification used in the publication that describes the structure. This does
not need to be numeric and is therefore of type character.
\item n_peptides: The number of peptides from one protein that were searched for within the
current structure.
\item n_peptides_in_structure: The number of peptides from one protein that were found within
the current structure.
\item percentage_covered_peptides: The percentage of all provided peptides that were at least
partially found in the structure.
}
}
\description{
Finds peptide positions in a PDB structure. Often positions of peptides in UniProt and a PDB
structure are different due to different lengths of structures. This function maps a peptide
based on its UniProt positions onto a PDB structure. This method is superior to sequence
alignment of the peptide to the PDB structure sequence, since it can also match the peptide if
there are truncations or mismatches. This function also provides an easy way to check if a
peptide is present in a PDB structure.
}
\examples{
\donttest{
# Create example data
peptide_data <- data.frame(
  uniprot_id = c("P0A8T7", "P0A8T7", "P60906"),
  peptide_sequence = c(
    "SGIVSFGKETKGKRRLVITPVDGSDPYEEMIPKWRQLNV",
    "NVFEGERVER",
    "AIGEVTDVVEKE"
  ),
  start = c(1160, 1197, 55),
  end = c(1198, 1206, 66)
)

# Find peptides in protein structure
peptide_in_structure <- find_peptide_in_structure(
  peptide_data = peptide_data,
  peptide = peptide_sequence,
  start = start,
  end = end,
  uniprot_id = uniprot_id
)

head(peptide_in_structure, n = 10)
}
}
