#' Read a flat binary table and run quant_es_core
#'
#' @description
#' Reads a TSV, CSV or other delimited file where:
#' * the first column is a variant identifier (unless removed)
#' * remaining columns contain binary 0, 1, or NA evidence indicators.
#'
#' @param path Path to a text file.
#' @param sep Field separator (default tab).
#' @param header Whether the file has a header.
#' @param variant_col Column name containing the variant IDs.  
#'        If NULL and no such column exists, sequential IDs will be created.
#'
#' @return A standard quantbayes result list.
#'
#' @examples
#' tmp <- tempfile(fileext = ".tsv")
#' write.table(core_test_data, tmp, sep = "\t", quote = FALSE, row.names = FALSE)
#' res <- quant_es_from_binary_table(tmp)
#' res$global
#'
#' @export
quant_es_from_binary_table <- function(path,
                                       sep = "\t",
                                       header = TRUE,
                                       variant_col = NULL) {
  
  df <- utils::read.table(
    path,
    sep = sep,
    header = header,
    stringsAsFactors = FALSE,
    check.names = FALSE
  )
  
  # Identify variant column
  if (!is.null(variant_col) && variant_col %in% names(df)) {
    ids <- df[[variant_col]]
    evid <- df[, setdiff(names(df), variant_col), drop = FALSE]
  } else {
    # If no variant column provided, auto-generate
    ids <- seq_len(nrow(df))
    evid <- df
  }
  
  # Validate evidence columns: must be 0, 1, or NA after numeric coercion
  is_binary_column <- function(x) {
    suppressWarnings({
      xn <- as.numeric(x)
      all(is.na(xn) | xn %in% c(0, 1))
    })
  }
  
  bad_cols <- names(evid)[!vapply(evid, is_binary_column, logical(1))]
  
  if (length(bad_cols) > 0) {
    stop(
      "Evidence columns must contain only 0, 1 or NA. ",
      "Invalid: ", paste(bad_cols, collapse = ", ")
    )
  }
  
  # Convert to matrix
  x <- suppressWarnings(apply(evid, 2, as.numeric))
  x <- as.matrix(x)
  rownames(x) <- ids
  
  quant_es_core(x)
}
