#' Install quartify RStudio Snippets
#'
#' Installs useful RStudio snippets for working with quartify. These snippets
#' help you quickly insert common structures when writing R scripts that will
#' be converted to Quarto documents.
#'
#' The following snippets are installed:
#'
#' - **header**: Insert a standard R script header with Title, Author, Date, and Description
#' - **callout**: Insert a Quarto callout structure
#' - **mermaid**: Insert a Mermaid diagram chunk
#' - **tabset**: Insert a tabset structure
#'
#' @details
#' By default the snippets are written to a temporary file (inside `tempdir()`)
#' to comply with CRAN's restriction on writing to the user's filespace during
#' examples, vignettes, and tests. To install in your actual RStudio snippets
#' file, provide the explicit path via `path`, for example:
#' - Windows: `%APPDATA%/RStudio/snippets/r.snippets`
#' - Mac/Linux: `~/.config/rstudio/snippets/r.snippets`
#'
#' If you already have custom snippets, this function will append the quartify
#' snippets to your existing file. If quartify snippets were previously installed,
#' they will be automatically removed and replaced with the new version.
#'
#' The function will automatically open the snippets file in RStudio if available.
#' Simply save the file (Ctrl+S / Cmd+S) to reload the snippets immediately
#' without restarting RStudio. Then type the snippet name (e.g., `header`)
#' and press Tab to insert the template.
#'
#' @param backup Logical. If TRUE (default), creates a backup of your existing
#'   snippets file before modifying it.
#' @param path Character. Path for the snippets file. Defaults to writing in
#'   `tempdir()` to comply with CRAN policies. Provide an explicit path when you
#'   intentionally want to install into your RStudio snippets directory.
#'
#' @return Invisibly returns the path to the snippets file.
#'
#' @export
#'
#' @examples
#' if (interactive()) {
#'   # Install quartify snippets to RStudio
#'   install_quartify_snippets()
#'
#'   # Install without backup
#'   install_quartify_snippets(backup = FALSE)
#' }
#'
#' # For testing: install to temp directory
#' temp_snippets <- file.path(tempdir(), "r.snippets")
#' install_quartify_snippets(path = temp_snippets)
install_quartify_snippets <- function(backup = TRUE, path = file.path(tempdir(), "r.snippets")) {
  # Define the snippets content
  snippets <- '
# quartify snippets

snippet header
	# Title : ${1:Title}
	#
	# Author : ${2:Author}
	#
	# Date : ${3:`r Sys.Date()`}
	#
	# Description : ${4:Description}
	#

snippet callout
	# callout-${1:note} - ${2:Title}
	# ${0}

snippet mermaid
	#| mermaid
	#| eval: true
	# ${0}

snippet tabset
	# tabset
	# tab - ${1:Tab Title}
	# ${0}
'

  # Robust detection of RStudio snippets file path
  get_rstudio_snippets_path <- function() {
    if (!is.null(path)) return(normalizePath(path, mustWork = FALSE))
    # Windows
    if (.Platform$OS.type == "windows") {
      appdata <- Sys.getenv("APPDATA")
      if (nzchar(appdata)) {
        return(file.path(appdata, "RStudio", "snippets", "r.snippets"))
      }
    }
    # XDG (Linux)
    xdg <- Sys.getenv("XDG_CONFIG_HOME")
    if (nzchar(xdg)) {
      p <- file.path(xdg, "rstudio", "snippets", "r.snippets")
      if (file.exists(dirname(p)) || dir.create(dirname(p), recursive = TRUE, showWarnings = FALSE)) return(p)
    }
    # RStudio Workbench/Server/Desktop (Linux/Mac)
    home <- Sys.getenv("HOME")
    if (nzchar(home)) {
      # Modern config path
      p1 <- file.path(home, ".config", "rstudio", "snippets", "r.snippets")
      if (file.exists(dirname(p1)) || dir.create(dirname(p1), recursive = TRUE, showWarnings = FALSE)) return(p1)
      # Legacy path
      p2 <- file.path(home, ".R", "snippets", "r.snippets")
      if (file.exists(dirname(p2)) || dir.create(dirname(p2), recursive = TRUE, showWarnings = FALSE)) return(p2)
    }
    stop("Unable to determine RStudio snippets file path.")
  }

  snippets_file <- get_rstudio_snippets_path()
  snippets_dir <- dirname(snippets_file)
  if (!dir.exists(snippets_dir)) dir.create(snippets_dir, recursive = TRUE)

  # Read existing file, backup, and remove old quartify block
  if (file.exists(snippets_file)) {
    existing_content <- readLines(snippets_file, warn = FALSE)
    # Backup existing file
    if (backup) {
      backup_file <- paste0(snippets_file, ".backup.", format(Sys.time(), "%Y%m%d_%H%M%S"))
      file.copy(snippets_file, backup_file, overwrite = TRUE)
      message("Created backup: ", backup_file)
    }
    # Remove old quartify block if exists
    start <- grep("# quartify snippets", existing_content, fixed = TRUE)
    if (length(start) > 0) {
      end <- which(seq_along(existing_content) > start[1] & existing_content == "")
      if (length(end) > 0) {
        end <- end[1] - 1
      } else {
        end <- length(existing_content)
      }
      existing_content <- existing_content[-c(start[1]:end)]
    }
    # Write back content (preserves other snippets)
    writeLines(existing_content, snippets_file)
    # Add blank line if needed
    if (length(existing_content) > 0 && existing_content[length(existing_content)] != "") {
      cat("\n", file = snippets_file, append = TRUE)
    }
    cat(snippets, file = snippets_file, append = TRUE)
    message("quartify snippets installed/updated in: ", snippets_file)
  } else {
    cat(snippets, file = snippets_file)
    message("Created new snippets file: ", snippets_file)
  }

  message("\nSnippets installed successfully!")
  message("If snippets do not appear, open the file below in RStudio and save it to reload snippets immediately.")
  message(snippets_file)
  message("\nAvailable snippets:")
  message("  - header  : R script header template")
  message("  - callout : Quarto callout structure")
  message("  - mermaid : Mermaid diagram chunk")
  message("  - tabset  : Tabset structure")
  message("\nType the snippet name and press Tab to use it.")

  # Open in RStudio if available
  if (requireNamespace("rstudioapi", quietly = TRUE) && rstudioapi::isAvailable()) {
    try(rstudioapi::navigateToFile(snippets_file), silent = TRUE)
    message("The snippets file has been opened in RStudio. Save it to reload snippets immediately.")
  }

  invisible(snippets_file)
}
