% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/describes_functions.R
\name{mean_change}
\alias{mean_change}
\title{Mean Change Across Two Timepoints (dependent two-samples t-test)}
\usage{
mean_change(
  pre,
  post,
  standardizer = "pre",
  d.ci.type = "unbiased",
  ci.level = 0.95,
  check = TRUE
)
}
\arguments{
\item{pre}{numeric vector of the variable at the pre-timepoint.}

\item{post}{numeric vector of the variable at the post-timepoint. The
elements must correspond to the same cases in \code{pre} as pairs by
position. Thus, the length of \code{post} must be the same as \code{pre}.
Note, missing values in \code{post} are expected and handled with listwise
deletion.}

\item{standardizer}{chararacter vector of length 1 specifying what to use for
standardization when computing the standardized mean difference (i.e.,
Cohen's d). There are three options: 1. "pre" for the standard deviation of
the pre-timepoint, 2. "pooled" for the pooled standard deviation of the
pre-timepoint and post-timepoint, 3. "change" for the standard deviation of
the change score (post - pre). The default is "pre", which I believe makes
the most theoretical sense (see Cumming, 2012); however, "change" is the
traditional choice originally proposed by Jacob Cohen (Cohen, 1988).}

\item{d.ci.type}{character vector of lenth 1 specifying how to compute the
confidence interval (and standard error) of the standardized mean
difference. There are currently two options: 1. "unbiased" which calculates
the unbiased standard error of Cohen's d based on the formulas in
Viechtbauer (2007). If \code{standardizer} = "pre" or "pooled", then
equation 36 from Table 2 is used. If \code{standardizer} = "change", then
equation 25 from Table 1 is used. A symmetrical confidence interval is then
calculated based on the standard error. 2. "classic" which calculates the
confidence interval of Cohen's d based on the confidence interval of the
mean change itself. The lower and upper confidence bounds are divided by
the \code{standardizer}. Technically, this confidence interval is biased
due to not taking into account the uncertainty of the \code{standardizer}.
No standard error is calculated for this option and NA is returned for
"d_se" in the return object.}

\item{ci.level}{double vector of length 1 specifying the confidence level.
\code{ci.level} must range from 0 to 1.}

\item{check}{logical vector of length 1 specifying whether the input
arguments should be checked for errors. For example, checking whether
\code{post} is the same length as \code{pre}. This is a tradeoff between
computational efficiency (FALSE) and more useful error messages (TRUE).}
}
\value{
list of numeric vectors containing statistical information about the
  mean change: 1) nhst = dependent two-samples t-test stat info in a numeric vector,
  2) desc = descriptive statistics stat info in a numeric vector, 3) std =
  standardized mean difference stat info in a numeric vector

1) nhst = dependent two-samples t-test stat info in a numeric vector

\describe{
   \item{est}{mean change estimate (i.e., post - pre)}
   \item{se}{standard error}
   \item{t}{t-value}
   \item{df}{degrees of freedom}
   \item{p}{two-sided p-value}
   \item{lwr}{lower bound of the confidence interval}
   \item{upr}{upper bound of the confidence interval}
}

2) desc = descriptive statistics stat info in a numeric vector

\describe{
   \item{mean_post}{mean of the post variable}
   \item{mean_pre}{mean of the pre variable}
   \item{sd_post}{standard deviation of of the post variable}
   \item{sd_pre}{standard deviation of the pre variable}
   \item{n}{sample size of the change score}
   \item{r}{Pearson correlation between the pre and post variables}
}

3) std = standardized mean difference stat info in a numeric vector

\describe{
   \item{d_est}{Cohen's d estimate}
   \item{d_se}{Cohen's d standard error}
   \item{d_lwr}{Cohen's d lower bound of the confidence interval}
   \item{d_upr}{Cohen's d upper bound of the confidence interval}
}
}
\description{
\code{mean_change} tests for mean change across two timepoints with a
dependent two-samples t-test. The function also calculates the descriptive
statistics for the timepoints and the standardized mean difference (i.e.,
Cohen's d) based on either the standard deviation of the pre-timepoint,
pooled standard deviation of the pre-timepoint and post-timepoint, or the
standard deviation of the change score (post - pre). \code{mean_change} is
simply a wrapper for \code{\link[stats]{t.test}} plus some extra
calculations.
}
\details{
\code{mean_change} calculates the mean change as \code{post} - \code{pre}
such that increases over time have a positive mean change estimate and
decreases over time have a negative mean change estimate. This would be as if
the post-timepoint was \code{x} and the pre-timepoint was \code{y} in
\code{t.test(paired = TRUE)}.
}
\examples{

# dependent two-sample t-test
mean_change(pre = mtcars$"disp", post = mtcars$"hp") # standardizer = "pre"
mean_change(pre = mtcars$"disp", post = mtcars$"hp", d.ci.type = "classic")
mean_change(pre = mtcars$"disp", post = mtcars$"hp", standardizer = "pooled")
mean_change(pre = mtcars$"disp", post = mtcars$"hp", ci.level = 0.99)
mean_change(pre = mtcars$"hp", post = mtcars$"disp",
   ci.level = 0.99) # note, when flipping pre and post, the cohen's d estimate
   # changes with standardizer = "pre" because the "pre" variable is different.
   # This does not happen for standardizer = "pooled" or "change". For example...
mean_change(pre = mtcars$"disp", post = mtcars$"hp", standardizer = "pooled")
mean_change(pre = mtcars$"hp", post = mtcars$"disp", standardizer = "pooled")
mean_change(pre = mtcars$"disp", post = mtcars$"hp", standardizer = "change")
mean_change(pre = mtcars$"hp", post = mtcars$"disp", standardizer = "change")

# same as intercept-only regression with the change score
mean_change(pre = mtcars$"disp", post = mtcars$"hp")
lm_obj <- lm(hp - disp ~ 1, data = mtcars)
coef(summary(lm_obj))

}
\references{
Cohen, J. (1988). Statistical power analysis for the behavioral sciences,
   2nd ed. Hillsdale, NJ: Erlbaum.

   Cumming, G. (2012). Understanding the new statistics: Effect sizes,
   confidence intervals, and meta-analysis. New York, NY: Rouledge.

   Viechtbauer, W. (2007). Approximate confidence intervals for standardized
   effect sizes in the two-independent and two-dependent samples design.
   Journal of Educational and Behavioral Statistics, 32(1), 39-60.
}
\seealso{
\code{\link{means_change}} for multiple sets of prepost pairs of variables,
   \code{\link[stats]{t.test}} the workhorse for \code{mean_change},
   \code{\link{mean_diff}} for a independent two-samples t-test,
   \code{\link{mean_test}} for a one-sample t-test,
}
