% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_forest.R
\name{plot_forest}
\alias{plot_forest}
\title{Create a Forest Plot from an rbmi Pool Object}
\usage{
plot_forest(
  pool_obj,
  display = c("trt", "lsm"),
  ref_value = NULL,
  ci_level = NULL,
  arm_labels = NULL,
  title = NULL,
  text_size = 3.5,
  point_size = 3.5,
  show_pvalues = TRUE,
  font_family = NULL,
  panel_widths = NULL
)
}
\arguments{
\item{pool_obj}{A pooled analysis object of class \code{"pool"}, typically
obtained from \code{\link[rbmi:pool]{rbmi::pool()}} after calling \code{\link[=analyse_mi_data]{analyse_mi_data()}}.}

\item{display}{Character string specifying the display mode. \code{"trt"}
(the default) shows treatment differences across visits. \code{"lsm"} shows
LS mean estimates by treatment arm with color-coded points.}

\item{ref_value}{Numeric. The reference value for the vertical reference
line. Default is \code{0} for \code{display = "trt"} and \code{NULL} (no line) for
\code{display = "lsm"}. Set explicitly to override.}

\item{ci_level}{Numeric. Confidence level for CI labeling. If \code{NULL}
(the default), extracted from \code{pool_obj$conf.level}. Falls back to 0.95 if
neither is available.}

\item{arm_labels}{Named character vector with elements \code{"ref"} and \code{"alt"}
providing custom labels for the reference and treatment arms when
\code{display = "lsm"}. If \code{NULL} (the default), uses \code{"Reference"} and
\code{"Treatment"}.}

\item{title}{Optional character string for the plot title.}

\item{text_size}{Numeric. Text size for the table and p-value panels.
Default is 3.5.}

\item{point_size}{Numeric. Point size for the forest plot. Default is 3.5.}

\item{show_pvalues}{Logical. Whether to display the p-value panel on the
right side of the plot. Default is \code{TRUE}. Set to \code{FALSE} for a cleaner
two-panel layout without p-values.}

\item{font_family}{Optional character string specifying the font family for
all text in the plot. When \code{NULL} (default), uses ggplot2's default font
(typically sans-serif). Applied to all \code{geom_text} layers and the forest
panel theme.}

\item{panel_widths}{Optional numeric vector controlling the relative widths
of the plot panels. When \code{show_pvalues = TRUE}, must be length 3 (table,
forest, p-value panels). When \code{show_pvalues = FALSE}, must be length 2
(table, forest panels). When \code{NULL} (default), uses \code{c(3, 4, 1.5)} for
3-panel and \code{c(3, 5)} for 2-panel layouts.}
}
\value{
A patchwork/ggplot object that can be further customized using
\verb{& theme()} to modify all panels simultaneously.
}
\description{
Takes an rbmi pool object and produces a publication-quality, three-panel
forest plot using ggplot2 and patchwork. The plot displays treatment effect
point estimates with confidence interval whiskers, an aligned table panel
showing formatted estimates, and a p-value panel.
}
\details{
The function calls \code{\link[=tidy_pool_obj]{tidy_pool_obj()}} internally to parse the pool object,
then constructs a three-panel composition:
\itemize{
\item \strong{Left panel:} Visit labels and formatted estimate with CI text
\item \strong{Middle panel:} Forest plot with point estimates and CI whiskers
\item \strong{Right panel:} Formatted p-values
}

\strong{Display modes:}
\itemize{
\item \code{"trt"} -- Treatment differences with a reference line at zero
(or custom \code{ref_value}). Significant results (CI excludes reference)
are shown as filled circles; non-significant as open circles.
\item \code{"lsm"} -- LS mean estimates by treatment arm, color-coded using
the Okabe-Ito colorblind-friendly palette (blue for reference,
vermilion for treatment). Points are dodged vertically within
each visit.
}

\strong{Customization:} The returned patchwork object supports \verb{& theme()} for
applying theme changes to all panels. For example:
\code{plot_forest(pool_obj) & theme(text = element_text(size = 14))}.

\strong{Suggested dimensions for regulatory documents:} For A4 or US Letter page
sizes, \verb{width = 10, height = 3 + 0.4 * n_visits} (in inches) provides good
results when saving with \code{ggplot2::ggsave()}. For example, a 5-visit plot
works well at 10 x 5 inches.

\strong{Example output (treatment difference mode):}

\if{html}{\figure{plot_forest-trt.png}{options: width=700}}
}
\examples{
\donttest{
if (requireNamespace("ggplot2", quietly = TRUE) &&
    requireNamespace("patchwork", quietly = TRUE)) {
  library(rbmi)
  data("ADMI", package = "rbmiUtils")
  ADMI$TRT <- factor(ADMI$TRT, levels = c("Placebo", "Drug A"))
  ADMI$USUBJID <- factor(ADMI$USUBJID)
  ADMI$AVISIT <- factor(ADMI$AVISIT)

  vars <- set_vars(
    subjid = "USUBJID", visit = "AVISIT", group = "TRT",
    outcome = "CHG", covariates = c("BASE", "STRATA", "REGION")
  )
  method <- method_bayes(
    n_samples = 20,
    control = control_bayes(warmup = 20, thin = 1)
  )

  ana_obj <- analyse_mi_data(ADMI, vars, method, fun = ancova)
  pool_obj <- pool(ana_obj)

  # Treatment difference forest plot
  plot_forest(pool_obj, arm_labels = c(ref = "Placebo", alt = "Drug A"))

  # LSM display with custom panel widths
  plot_forest(
    pool_obj,
    display = "lsm",
    arm_labels = c(ref = "Placebo", alt = "Drug A"),
    title = "LS Mean Estimates by Visit",
    panel_widths = c(3, 5, 1.5)
  )
}
}

}
\seealso{
\itemize{
\item \code{\link[rbmi:pool]{rbmi::pool()}} for creating pool objects
\item \code{\link[=tidy_pool_obj]{tidy_pool_obj()}} for the underlying data transformation
\item \code{\link[=efficacy_table]{efficacy_table()}} for tabular presentation of the same data
\item \code{\link[=format_pvalue]{format_pvalue()}} for p-value formatting rules
\item \code{\link[=format_estimate]{format_estimate()}} for estimate with CI formatting
}
}
