\name{recluster.region}
\alias{recluster.region}
\title{
Consensus-based regionalisation from repeated row-order resampling
}
\description{
This function implements a consensus approach to regionalisation that is robust to ties and zero values in dissimilarity matrices, a common issue with turnover-based beta-diversity at small or intermediate spatial scales.

The input can be either (i) a community-by-site matrix (areas/sites in rows, species in columns), or (ii) any \code{\link[stats]{dist}} object. When a community matrix is provided, dissimilarities are computed using \code{\link{recluster.dist}} (optionally including a phylogeny for phylogenetic beta-diversity).

The core idea is to repeatedly permute the row order of the input, perform clustering on each permuted dissimilarity matrix, and then summarise, for each requested number of clusters \eqn{k}, how often pairs of areas are assigned to different clusters. This yields, for each \eqn{k}, a consensus dissimilarity matrix among areas, which is then clustered to obtain final regionalisation solutions. Each solution is evaluated using mean silhouette width and explained dissimilarity.
}
\usage{
recluster.region(mat, tr = 50, dist = "simpson", method = "ward.D2",
  members = NULL, phylo = NULL, mincl = 2, maxcl = 10,
  rettree = FALSE, retmat = FALSE, retmemb = FALSE)
}
\arguments{
  \item{mat}{
A community-by-site matrix (rows = areas/sites, columns = species; typically presence/absence) or a \code{\link[stats]{dist}} object.
}

  \item{tr}{
Number of random permutations (re-orderings) used to build the consensus.
}

  \item{dist}{
A beta-diversity index supported by \code{\link{recluster.dist}} (or a custom index via \code{\link[vegan]{designdist}} syntax). Not used when \code{mat} is a \code{dist} object.
}

  \item{method}{
Clustering method. Supported values are:
\describe{
  \item{\code{"pam"}}{Partitioning Around Medoids via \code{\link[cluster]{pam}}.}
  \item{\code{"diana"}}{Divisive hierarchical clustering via \code{\link[cluster]{diana}}.}
  \item{Any \code{\link[stats]{hclust}} method}{Agglomerative hierarchical clustering (e.g. \code{"ward.D2"}, \code{"average"}, \code{"complete"}).}
}
}

  \item{members}{
Optional vector of cluster sizes passed to \code{\link[stats]{hclust}}. Ignored when \code{method = "pam"} or \code{method = "diana"}.
}

  \item{phylo}{
An ultrametric rooted phylogenetic tree (with tip labels matching \code{colnames(mat)}) used to compute phylogenetic beta-diversity in \code{\link{recluster.dist}}.
}

  \item{mincl}{
Minimum number of clusters \eqn{k} to evaluate.
}

  \item{maxcl}{
Maximum number of clusters \eqn{k} to evaluate.
}

  \item{rettree}{
Logical. If \code{TRUE}, returns the final clustering objects for each \eqn{k}.
}

  \item{retmat}{
Logical. If \code{TRUE}, returns the consensus dissimilarity matrices used for the final clustering step.
}

  \item{retmemb}{
Logical. If \code{TRUE}, returns the raw membership array from all permutations.
}
}
\details{
For each permutation, a clustering is built on the permuted dissimilarity matrix and cut into \eqn{k} clusters for \eqn{k = mincl, \dots, maxcl}. For each \eqn{k}, the resulting membership assignments across permutations are converted into a consensus dissimilarity among areas (based on how frequently pairs of areas fall into different clusters). A final clustering is then applied to the consensus dissimilarity matrix to obtain a single regionalisation for each \eqn{k}.

Two diagnostics are provided:
\itemize{
  \item \strong{Mean silhouette width} (range -1 to 1), computed on the original dissimilarity matrix for the final grouping.
  \item \strong{Explained dissimilarity} (sensu Holt et al. 2013), computed by \code{\link{recluster.expl}} / \code{\link{recluster.expl.diss}} (note that it tends to increase as \eqn{k} increases).
}
}
\value{
A list with components:
\describe{
  \item{\code{grouping}}{
A matrix of cluster memberships (rows = areas/sites; columns correspond to \eqn{k = mincl, \dots, maxcl}).
  }
  \item{\code{solutions}}{
A matrix with columns \code{k}, \code{silh} (mean silhouette width), and \code{ex.diss} (explained dissimilarity).
  }
  \item{\code{tree}}{
Optional. A list of clustering objects for each \eqn{k} when \code{rettree = TRUE}.
  }
  \item{\code{matrices}}{
Optional. A 3D array of consensus dissimilarity matrices (one per \eqn{k}) when \code{retmat = TRUE}.
  }
  \item{\code{memb}}{
Optional. A 3D array of raw memberships from all permutations (areas \eqn{\times} permutations \eqn{\times} \eqn{k}) when \code{retmemb = TRUE}.
  }
}
}
\references{
Dapporto L. et al. (2015).
A new procedure for extrapolating turnover regionalization at mid-small spatial scales, tested on British butterflies.
\emph{Methods in Ecology and Evolution}, 6, 1287--1297.
}
\author{
Leonardo Dapporto
}
\examples{
data(dataisl)

# Example using a precomputed dissimilarity matrix:
set.seed(1)
simpson <- recluster.dist(dataisl)

#Alternatively with betapart
if (requireNamespace("betapart", quietly = TRUE)) {
  dsim <- betapart::beta.pair(dataisl)$beta.sim
}

turn_cl <- recluster.region(simpson, tr = 10, mincl = 2, maxcl = 4, rettree = TRUE)

# Plot the tree for k = 3 (i.e., the 2nd column if mincl=2):
plot(turn_cl$tree[[2]])

# Inspect cluster membership solutions:
turn_cl$grouping
}
\keyword{cluster}