##' Create a Redis connection.  This function is designed to be used
##' in other packages, and not directly by end-users.  However, it is
##' possible and safe to use.  See the [hiredis()] function for the
##' user friendly interface.
##'
##' This function creates a list of functions, appropriately bound to
##' a pointer to a Redis connection.  This is designed for package
##' authors to use so without having to ever deal with the actual
##' pointer itself (which cannot be directly manipulated from R
##' anyway).
##'
##' The returned list has elements, all of which are functions:
##'
##' * `config()`: The configuration information
##'
##' * `reconnect()`: Attempt reconnection of a connection that has
##'   been closed, through serialisation/deserialisation or through
##'   loss of internet connection.
##'
##' * `command(cmd)`: Run a Redis command.  See below for the format.
##'
##' * `pipeline(cmds)`: Run a pipeline of Redis commands.
##'
##' * `subscribe(channel, pattern, callback, envir)`: Subscribe to a
##'   channel or pattern specifying channels.  Here, `channel` must be
##'   a character vector, `pattern` a logical indicating if `channel`
##'   should be interpreted as a pattern, `callback` is a function to
##'   apply to each received message, returning `TRUE` when
##'   subscription should stop, and `envir` is the environment in
##'   which to evaluate `callback`.  See below.
##'
##' # Arbitrary commands with `command()`
##'
##' Redis releases new commands frequently, or it's possible that the
##' wrapper created by redux is too inflexible for your use case.  In
##' this situation you can use the `command()` method to send
##' arbitrary commands to the server and either use these unsupported
##' commands, or fundamentally change how they work.
##'
##' The `command` function takes a single unnamed argument, being a
##' list of commands.  The first element of this will always be the
##' name of a redis command (an uppercase string, such as `HMSET` or
##' `AUTH`) and subsequent arguments will be strings, raw vectors or
##' `NULL`.  Strings and raw vectors are passed as-is, while `NULL`
##' values are skipped over.
##'
##' **Spaces within strings are not interpreted as command
##' separators**.  So you cannot pass, for example
##'
##' ```
##' r$command(list("SET", "a b"))
##' ```
##'
##' and have redis interpret this as two arguments to `SET`.  You must
##' pass each argument as an element within the list
##'
##' ```
##' r$command(list("SET", "a", "b"))
##' ```
##'
##' Raw vectors can be useful for passing in serialised R objects, you
##' can use [object_to_bin()] and [bin_to_object()] to simplify this
##' process.
##'
##' ```
##' r$command(list("SET", "a", object_to_bin(mtcars)))
##' ```
##'
##' # Subscriptions
##'
##' The callback function must take a single argument; this will be
##' the received message with named elements `type` (which will be
##' message), `channel` (the name of the channel) and `value` (the
##' message contents).  If `pattern` was `TRUE`, then an additional
##' element `pattern` will be present (see the Redis docs).  The
##' callback must return `TRUE` or `FALSE`; this indicates if the
##' client should continue quit (i.e., `TRUE` means return control to
##' R, `FALSE` means keep going).
##'
##' Because the `subscribe` function is blocking and returns nothing,
##' so all data collection needs to happen as a side-effect of the
##' callback function.
##'
##' There is currently no way of interrupting the client while it is
##' aiting for a message.
##'
##' @title Create a Redis connection
##'
##' @param config Configuration parameters as generated by
##'   [redis_config()]
##'
##' @useDynLib redux, .registration = TRUE
##' @export
##' @examplesIf redux::redis_available()
##' con <- redis_connection()
##' con$command(list("PING"))
redis_connection <- function(config = redis_config()) {
  config <- redis_config(config)
  ptr <- redis_connect(config)
  ret <-
    list(
      config = function() {
        config
      },

      reconnect = function() {
        ptr <<- redis_connect(config)
        invisible()
      },

      command = function(cmd) {
        redis_command(ptr, cmd)
      },

      pipeline = function(cmds) {
        redis_pipeline(ptr, cmds)
      },

      subscribe = function(channel, pattern, callback, envir = parent.frame()) {
        redis_subscribe(ptr, channel, pattern, callback, envir)
      })
  attr(ret, "type") <- "redux"
  class(ret) <- "redis_connection"
  ret
}
