% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KrigingClass.R
\name{Kriging}
\alias{Kriging}
\title{Create an object with S3 class \code{"Kriging"} using
the \pkg{libKriging} library.}
\usage{
Kriging(
  y = NULL,
  X = NULL,
  kernel = NULL,
  regmodel = c("constant", "linear", "interactive", "none"),
  normalize = FALSE,
  optim = c("BFGS", "Newton", "none"),
  objective = c("LL", "LOO", "LMP"),
  parameters = NULL
)
}
\arguments{
\item{y}{Numeric vector of response values.}

\item{X}{Numeric matrix of input design.}

\item{kernel}{Character defining the covariance model:
\code{"exp"}, \code{"gauss"}, \code{"matern3_2"}, \code{"matern5_2"}.}

\item{regmodel}{Universal Kriging linear trend: \code{"constant"}, 
\code{"linear"}, \code{"interactive"}, \code{"quadratic"}.}

\item{normalize}{Logical. If \code{TRUE} both the input matrix
\code{X} and the response \code{y} in normalized to take
values in the interval \eqn{[0, 1]}.}

\item{optim}{Character giving the Optimization method used to fit
hyper-parameters. Possible values are: \code{"BFGS"},
\code{"Newton"} and \code{"none"}, the later simply keeping
the values given in \code{parameters}. The method
\code{"BFGS"} uses the gradient of the objective 
(note that \code{"BGFS10"} means 10 multi-start of BFGS). The method
\code{"Newton"} uses both the gradient and the Hessian of the
objective.}

\item{objective}{Character giving the objective function to
optimize. Possible values are: \code{"LL"} for the
Log-Likelihood, \code{"LOO"} for the Leave-One-Out sum of
squares and \code{"LMP"} for the Log-Marginal Posterior.}

\item{parameters}{Initial values for the hyper-parameters. When
provided this must be named list with elements \code{"sigma2"}
and \code{"theta"} containing the initial value(s) for the
variance and for the range parameters. If \code{theta} is a
matrix with more than one row, each row is used as a starting
point for optimization.}
}
\value{
An object with S3 class \code{"Kriging"}. Should be used
    with its \code{predict}, \code{simulate}, \code{update}
    methods.
}
\description{
The hyper-parameters (variance and vector of correlation ranges)
are estimated thanks to the optimization of a criterion given by
\code{objective}, using the method given in \code{optim}.
}
\examples{
f <- function(x) 1 - 1 / 2 * (sin(12 * x) / (1 + x) + 2 * cos(7 * x) * x^5 + 0.7)
set.seed(123)
X <- as.matrix(runif(10))
y <- f(X)
## fit and print
k <- Kriging(y, X, kernel = "matern3_2")
print(k)

x <- as.matrix(seq(from = 0, to = 1, length.out = 101))
p <- predict(k, x = x, return_stdev = TRUE, return_cov = FALSE)

plot(f)
points(X, y)
lines(x, p$mean, col = "blue")
polygon(c(x, rev(x)), c(p$mean - 2 * p$stdev, rev(p$mean + 2 * p$stdev)),
border = NA, col = rgb(0, 0, 1, 0.2))

s <- simulate(k, nsim = 10, seed = 123, x = x)

matlines(x, s, col = rgb(0, 0, 1, 0.2), type = "l", lty = 1)
}
\author{
Yann Richet \email{yann.richet@asnr.fr}
}
