% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shapeR.R
\docType{class}
\name{shapeR}
\alias{shapeR}
\alias{shapeR-class}
\title{shapeR}
\usage{
shapeR(project.path, info.file, ...)
}
\arguments{
\item{project.path}{The base project path where the images are stored}

\item{info.file}{The information file which store the information on the fish and otoliths. This is the base for the master.list}

\item{...}{Additional parameters to be passed to 'read.csv' for reading the info.file}
}
\value{
a \code{\linkS4class{shapeR}} object
}
\description{
Collection and analysis of otolith shape data

a shapeR class
}
\section{Slots}{

\describe{
\item{\code{project.path}}{Path to the project where the images are stored}

\item{\code{info.file}}{Info file containing fish and otolith information}

\item{\code{master.list.org}}{The contents of the \code{info.file}}

\item{\code{master.list}}{The contents of the \code{info.file} with added shape parameters and descriptors}

\item{\code{outline.list.org}}{A list of all the original otolith outlines}

\item{\code{outline.list}}{A list of all the otolith outlines. It returns a list of smoothed if contour smoothing (using\code{smoothout}) has been conducted.}

\item{\code{filter}}{A logical vector selecting the otoliths used for analysis}

\item{\code{wavelet.coef.raw}}{The wavelet coefficients for all the otolith outlines}

\item{\code{wavelet.coef}}{The wavelet coefficients after aligning with the \code{info.file}. The data is generated when \link{enrich.master.list} is run}

\item{\code{wavelet.coef.std}}{The standardized wavelet coefficients. The data is generated when \link{stdCoefs} is run}

\item{\code{wavelet.coef.std.removed}}{The index of the removed wavelet coefficients after standardization. The data is generated when \link{stdCoefs} is run}

\item{\code{fourier.coef.raw}}{The Fourier coefficients for all the otolith outlines}

\item{\code{fourier.coef}}{The Fourier coefficients for after aligning with the info file. The data is generated when \link{enrich.master.list} is run}

\item{\code{fourier.coef.std}}{The standardized Fourier coefficients. The data is generated when \link{stdCoefs} is run}

\item{\code{fourier.coef.std.removed}}{The index of the removed Fourier coefficents after standardization. The data is generated when \link{stdCoefs} is run}

\item{\code{shape.coef.raw}}{The uncalibrated shape measurements for all the otoliths. The shape parameters are: otolith.area, otolith.length, otolith.width, otolith.perimeter}

\item{\code{shape.coef}}{The shape measurements for after aligning with the info file. The shape parameters have been calibrated using the calibration parameter as registered in the datafile as the column 'cal'.}

\item{\code{shape.std}}{The standardized shape measurements. The data is generated when \link{stdCoefs} is run}

\item{\code{shape.std.removed}}{The index of the removed shape measurements after standardization. The data is generated when \link{stdCoefs} is run}
}}

\examples{

\dontrun{

# This example has two sections: (1) Demonstration of how a shapeR object
# is analyzed and (2) How to create a shapeR object from an archive of
# image files.

#-----------------------------------------
# Section 1: Analyzing a shapeR object

data(shape)

#Standardize coefficients
shape = stdCoefs(shape,"pop","length_cm")

#Visualize Wavelet and Fourier coefficients
plotWavelet(shape,level=5,class.name= "pop",useStdcoef=TRUE)
plotFourier(shape,class.name= "pop",useStdcoef=TRUE)

#Examine the mean shapes
plotWaveletShape(shape, "pop",show.angle = TRUE,lwd=2,lty=1)
plotFourierShape(shape, "pop",show.angle = TRUE,lwd=2,lty=1)

#Canonical analysis
library(vegan)
cap.res = capscale(getStdWavelet(shape) ~ getMasterlist(shape)$pop)
anova(cap.res)

#Visualize the canonical scores
eig=eigenvals(cap.res,constrained=TRUE)
eig.ratio = eig/sum(eig)

cluster.plot(scores(cap.res)$sites[,1:2],getMasterlist(shape)$pop
,plotCI=TRUE
,xlab=paste("CAP1 (",round(eig.ratio[1]*100,1),"\%)",sep="")
,ylab=paste("CAP2 (",round(eig.ratio[2]*100,1),"\%)",sep="")
,main="Canonical clustering"
)

#Only analyze Icelandic and Norwegian samples
shape = setFilter(shape, getMasterlist(shape, useFilter = FALSE)$pop \%in\% c("NO","IC"))

#Classifier on standardized wavelet
lda.res.w = lda(getStdWavelet(shape),getMasterlist(shape)$pop,CV=TRUE)
ct.w = table(getMasterlist(shape)$pop,lda.res.w$class)

diag(prop.table(ct.w, 1))

# Total percent correct
sum(diag(prop.table(ct.w)))

cap.res = capscale(getStdWavelet(shape) ~ getMasterlist(shape)$pop)
anova(cap.res)

#Classifier on canoncial values
lda.res.w = lda(scores(cap.res)$sites,getMasterlist(shape)$pop,CV=TRUE)
ct.w = table(getMasterlist(shape)$pop,lda.res.w$class)

diag(prop.table(ct.w, 1))

# Total percent correct
sum(diag(prop.table(ct.w)))

#-----------------------------------------
# Section 2: Creating a shapeR object from image files

# The following example requires the user to download an archive of JPEG
# files from https://github.com/lisalibungan/shapeR/
# place the ShapeAnalysis directory inside the working directory.

shape = shapeR("~/ShapeAnalysis/","FISH.csv")
shape = detect.outline(shape,write.outline.w.org = TRUE)

shape = generateShapeCoefficients(shape)
shape = enrich.master.list(shape)
}
}
\references{
Libungan LA and Palsson S (2015) ShapeR: An R Package to Study Otolith Shape Variation among Fish Populations. PLoS ONE 10(3): e0121102. \url{https://journals.plos.org/plosone/article?id=10.1371/journal.pone.0121102}
}
\seealso{
\url{https://github.com/lisalibungan/shapeR}

\code{\link{plotWavelet}}

\code{\link{plotFourier}}

\code{\link{plotWaveletShape}}

\code{\link{plotFourierShape}}

\code{\link[vegan]{capscale}}

\code{\link{cluster.plot}}

\code{\link{setFilter}}

\code{\link[MASS]{lda}}

\code{\link{detect.outline}}

\code{\link{generateShapeCoefficients}}

\code{\link{enrich.master.list}}
}
\author{
Lisa Anne Libungan & Snaebjorn Palsson
}
