#' This class represents an SQL unary operator.
#'
#' Used to generate SQL expressions involving an unary operator like in
#' "NOT flag".
#'
#' @examples
#' # To generate "NOT flag":
#' ExprUnaryOp$new("not", ExprField$new("flag"))
#'
#' @import R6
#' @include ExprComp.R
#' @export
ExprUnaryOp <- R6::R6Class("ExprUnaryOp",
  inherit = ExprComp,
  public = list(

    #' @description
    #' Initializer.
    #' @param op The unary operator, as a string.
    #' @param expr An Expr instance.
    #' @param ... Arguments to pass to parent class.
    #' @return Nothing.
    initialize = function(op, expr, ...) {
      super$initialize(...)
      chk::chk_is(expr, "Expr")
      chk::chk_string(op)

      private$op <- op
      private$expr <- expr

      return(invisible(NULL))
    },

    #' @description
    #' Generates the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list()
      if (private$paren) {
        tokens <- c(tokens, .lparen)
      }
      tokens <- c(
        tokens,
        make_left_op(private$op),
        private$expr$getTokens()
      )
      if (private$paren) {
        tokens <- c(tokens, .rparen)
      }
      return(tokens)
    }
  ),
  private = list(
    op = NULL,
    expr = NULL
  )
)
