% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plumage.R
\name{plumage}
\alias{plumage}
\title{Identify Chronic Conditions Using ICD-10-AM U-Codes}
\usage{
plumage(df, icd_column, prefix = NULL, decimal = TRUE, drop_eggs = FALSE)
}
\arguments{
\item{df}{A data frame containing hospitalization records}

\item{icd_column}{Character string specifying the name of the column containing ICD-10-AM codes}

\item{prefix}{Optional character string to prefix all output column names (default: NULL)}

\item{decimal}{Logical indicating whether to match U-codes with decimal points (TRUE, default)
or without decimal points (FALSE). When TRUE, matches "U78.1" format; when FALSE, matches "U781" format.}

\item{drop_eggs}{Logical indicating whether to drop individual condition columns and retain only
summary columns. Default is FALSE.}
}
\value{
Returns the input data frame with additional columns:
* Binary indicators (0/1) for each chronic condition, optionally prefixed (unless drop_eggs = TRUE)
* total_conditions: Sum of all identified conditions
* total_metabolic_conditions: Sum of metabolic/endocrine conditions
* total_mental_health_conditions: Sum of mental health conditions
* total_neurological_conditions: Sum of neurological conditions
* total_cardiovascular_conditions: Sum of cardiovascular conditions
* total_respiratory_conditions: Sum of respiratory conditions
* total_gastrointestinal_conditions: Sum of gastrointestinal conditions
* total_musculoskeletal_conditions: Sum of musculoskeletal conditions
* total_renal_conditions: Sum of renal conditions
* total_congenital_conditions: Sum of congenital conditions
* conditions_category: Factor with levels "0", "1", "2", "3+" based on total_conditions
}
\description{
Analyzes a hospitalization dataset to identify chronic conditions based on ICD-10-AM U-codes.
Like identifying a bird by its distinctive plumage (feathers), this function identifies patients
by their chronic condition patterns. Creates binary indicators for each condition and calculates
total condition counts by category.
}
\details{
This function identifies chronic conditions from ICD-10-AM U-codes (Australian modification codes
for chronic conditions). The function recognizes the following conditions:

**Metabolic/Endocrine:**
* U78.1: Obesity
* U78.2: Cystic fibrosis

**Mental Health:**
* U79.1: Dementia
* U79.2: Schizophrenia
* U79.3: Depression
* U79.4: Intellectual/developmental disability

**Neurological:**
* U80.1: Parkinson's disease
* U80.2: Multiple sclerosis
* U80.3: Epilepsy
* U80.4: Cerebral palsy
* U80.5: Paralysis

**Cardiovascular:**
* U82.1: Ischaemic heart disease
* U82.2: Heart failure
* U82.3: Hypertension

**Respiratory:**
* U83.1: Emphysema
* U83.2: COPD
* U83.3: Asthma
* U83.4: Bronchiectasis
* U83.5: Respiratory failure

**Gastrointestinal:**
* U84.1: Crohn's disease
* U84.2: Ulcerative colitis
* U84.3: Liver failure

**Musculoskeletal:**
* U86.1: Rheumatoid arthritis
* U86.2: Osteoarthritis
* U86.3: Systemic lupus erythematosus
* U86.4: Osteoporosis

**Renal:**
* U87.1: Chronic kidney disease

**Congenital:**
* U88.1: Spina bifida
* U88.2: Down syndrome

The function searches for these codes within the specified ICD column and creates binary
indicators for each condition. It also calculates summary measures including total conditions
overall and by disease category.

Note: Cystic fibrosis (U78.2) is counted in both metabolic and respiratory categories.
}
\examples{
# Create sample hospitalization data
hospital_data <- data.frame(
  patient_id = 1:4,
  icd_codes = c(
    "K29.70",
    "U78.1, U83.2, U82.3",
    "U79.3, U83.3",
    "U80.1, U86.2"
  )
)

# Identify chronic conditions with decimal format (default)
results1 <- plumage(hospital_data, "icd_codes")

# View category summaries
results1[, c("patient_id", "total_conditions",
             "total_cardiovascular_conditions",
             "total_respiratory_conditions")]

# Identify chronic conditions without decimal format
results2 <- plumage(hospital_data, "icd_codes", decimal = FALSE)

# Identify chronic conditions with prefix
results3 <- plumage(hospital_data, "icd_codes", prefix = "chronic_")

# Keep only summary columns, drop individual conditions
results4 <- plumage(hospital_data, "icd_codes", drop_eggs = TRUE)

}
