#' STATIS DUAL Analysis
#'
#' Implementation of the STATIS DUAL method for the joint analysis of multiple tables
#' that share the same variables. This approach allows evaluating the common structure
#' between tables (interstructure), building a compromise (weighted average of structures),
#' and analyzing the trajectories of variables across the tables.
#'
#' @param tables A list of matrices or data frames with the same columns (variables). Each element represents a "table".
#' @param labels.tables A vector of length equal to the number of tables, used to name the tables in the results. If \code{NULL}, labels like \code{("T1", "T2", ...)} are auto-generated.
#'
#' @return A list with the following elements:
#' \item{labels.tables}{Vector with the table labels}
#' \item{interstructure}{K x 2 matrix with the coordinates of the tables in the interstructure space}
#' \item{supervariables}{p x 2 matrix with the coordinates of the variables in the compromise}
#' \item{trajectories}{List of p K x 2 matrices, one per variable, showing its trajectory across the tables}
#' \item{vars.names}{Names of the variables (common columns)}
#' \item{S}{Interstructure similarity matrix}
#' \item{R}{List of R matrices for each table}
#' \item{Comp}{Compromise matrix (weighted combination of R matrices)}
#' \item{eigenvalues.compromise}{Eigenvalues of the compromise (inertia per axis)}
#' \item{eigenvectors.compromise}{Eigenvectors of the compromise}
#' \item{beta.weights}{Weights of each table in the construction of the compromise}
#'
#' @details The STATIS DUAL method allows:
#' \itemize{
#'   \item Evaluating structural coherence across multiple tables using the interstructure
#'   \item Constructing a representative compromise of the set of tables
#'   \item Analyzing the behavior of the variables across the set (trajectories)
#' }
#'
#' Internally, the tables are centered and normalized considering uniform observation weights.
#' The R matrices capture the internal structure of each table. The interstructure is based on
#' scalar products between these matrices.
#'
#' @seealso \code{\link{plot.statis.dual.circle}}, \code{\link{plot.statis.dual.trajectories}}
#'
#' @examples
#' data(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98)
#' labels = c("95","96","97","98")
#'
#' res <- statis.dual(list(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98), labels.tables = labels)
#'
#' # How to use res
#' t <- ggplot2::ggtitle("Correlation (all variables)")
#' plot.statis.dual.circle(list(res$supervariables), labels = row.names(res$supervariables)) + t
#'
#' t <- ggplot2::ggtitle("Trajectories (all variables)")
#' plot.statis.dual.trajectories(res$vars.names, res$trajectories, res$labels.tables) + t
#'
#' @export
statis.dual <- function(tables, labels.tables = NULL) {
  # labels.tables: vector of length K to label each table (e.g., years, sites, treatments, etc.)

  # ----------------------------------------------------------------------
  # Input validation
  # tables must be a list of numeric matrices / data.frames with same columns
  # ----------------------------------------------------------------------
  stopifnot(is.list(tables), length(tables) >= 2)

  # Convert all data frames to matrices
  tables <- lapply(tables, function(x) if (is.data.frame(x)) as.matrix(x) else x)

  K <- length(tables) # Number of tables
  p <- ncol(tables[[1]]) # Number of variables (columns)
  stopifnot(all(vapply(tables, ncol, 1L) == p))
  if (is.null(labels.tables)) labels.tables <- paste0("T", seq_len(K)) # If names not provided, assign generic (T1, T2, ...TN)

  # ----------------------------------------------------------------------
  # Metric matrices and weights
  # M : variable metric (identity here)
  # Dk: weight for each table (uniform here, 1/K)
  # d : vector of row weights per table (uniform for each row)
  # ----------------------------------------------------------------------
  M <- diag(p)
  Dk <- rep(1 / K, K)
  d  <- lapply(tables, function(df) rep(1 / nrow(df), nrow(df)))

  # ----------------------------------------------------------------------
  # 1) CENTERING each table
  # Produces a p x n matrix per table.
  # ----------------------------------------------------------------------
  ZCT <- lapply(seq_len(K), function(j) {
    X  <- tables[[j]]
    dj <- d[[j]]
    mu <- as.numeric(crossprod(dj, X)) # Weighted mean of columns
    t(X) - matrix(mu, nrow = p, ncol = nrow(X)) # Transposed centered table
  })

  # ----------------------------------------------------------------------
  # 2) NORMALIZATION using the row weights
  # Produces ZCR (centered + normalized)
  # ----------------------------------------------------------------------
  ZCR <- lapply(seq_len(K), function(j) normalizes(ZCT[[j]], d[[j]]))

  # ----------------------------------------------------------------------
  # 3) Compute R matrices:
  # These capture the structure of each table using the chosen metrics.
  # ----------------------------------------------------------------------
  R <- lapply(seq_len(K), function(k) {
    Ak <- ZCR[[k]]; dk <- d[[k]]
    Ak %*% diag(dk) %*% t(Ak)
  })

  # ----------------------------------------------------------------------
  # 4) Compute RM (R matrices multiplied by variable metric M)
  # 5) Build S (similarity matrix between tables) S[k,s] = trace( RM_k * RM_s )
  # ----------------------------------------------------------------------
  RM <- lapply(R, function(Rk) Rk %*% M)
  S <- matrix(NA_real_, nrow = K, ncol = K)
  for (k in seq_len(K)) for (s in seq_len(K)) S[k, s] <- sum(RM[[k]] * RM[[s]])
  dimnames(S) <- list(labels.tables, labels.tables)

  # ----------------------------------------------------------------------
  # Interstructure Analysis (Normalized)
  # ----------------------------------------------------------------------
  Aux <- diag(1 / sqrt(diag(S))) # normalization operator
  Sn <- Aux %*% S %*% Aux # normalized S
  W   <- diag(Dk) # diagonal table weights

  E1  <- eigen(t(W %*% Sn))
  vlp1 <- E1$values # eigenvalues
  vcp1 <- t(E1$vectors) # eigenvectors
  vcpn <- normalizes(vcp1, Dk) # normalized eigenvectors
  H1   <- t(diag(sqrt(vlp1)) %*% vcpn) # contains first two dimensions of interstructure (principal coordinates)

  interstructure <- as.matrix(H1)[, 1:2, drop = FALSE]
  rownames(interstructure) <- labels.tables
  colnames(interstructure) <- c("x","y")

  # ----------------------------------------------------------------------
  # Compromise and Decomposition
  # ----------------------------------------------------------------------
  Axx_mat <- diag(1 / diag(S))
  M_ax <- diag(Dk) %*% (Axx_mat %*% S)
  E_ax <- eigen(M_ax, symmetric = FALSE)
  vcp1_ax <- t(E_ax$vectors)
  beta <- Re(vcp1_ax[1,]); beta <- beta / sum(beta) # normalize weights

  # Compromise matrix
  Comp <- Reduce(`+`, Map(function(Mi, w) w * Mi, R, beta))
  E_comp <- eigen(Comp, symmetric = TRUE)
  vlpc   <- Re(E_comp$values)
  vcpc   <- t(E_comp$vectors)

  # ----------------------------------------------------------------------
  # Supervariables (global component coordinates)
  # For the first two compromise axes
  # ----------------------------------------------------------------------
  supervariables <- cbind(sqrt(vlpc[1]) * as.numeric(vcpc[1, ]),
                          sqrt(vlpc[2]) * as.numeric(vcpc[2, ]))
  colnames(supervariables) <- c("x","y")
  rownames(supervariables) <- colnames(tables[[1]])
  vars.names <- rownames(supervariables)

  # ----------------------------------------------------------------------
  # Trajectories of each variable across tables
  # ----------------------------------------------------------------------
  A <- t( sweep(vcpc[1:2, , drop = FALSE], 1, sqrt(vlpc[1:2]), FUN = "/") )

  # coordvar[k, i, dim] = K x p x 2 array
  coordvar <- array(NA_real_, dim = c(K, p, 2),
                    dimnames = list(labels.tables, vars.names, c("Dim1","Dim2")))
  for (k in seq_len(K)) {
    Mki <- beta[k] * ( R[[k]] %*% A )
    coordvar[k, , 1] <- Mki[, 1]
    coordvar[k, , 2] <- Mki[, 2]
  }

  # Convert trajectories into a list of p matrices (each K x 2)
  trajectories <- lapply(seq_len(p), function(i) cbind(coordvar[, i, 1], coordvar[, i, 2]))
  names(trajectories) <- vars.names

  list(
    labels.tables = labels.tables,
    interstructure  = interstructure,     # K x 2 (x,y)
    supervariables   = supervariables,    # p x 2 (x,y)
    trajectories     = trajectories,      # list of p elements; each K x 2
    vars.names     = vars.names,
    S = S, R = R, Comp = Comp,
    eigenvalues.compromise = vlpc,
    eigenvectors.compromise = vcpc,
    beta.weights = beta
  )
}
