#' @title Dynamic branching over output or input files.
#' @export
#' @family Dynamic branching over files
#' @description Dynamic branching over output or input files.
#'   [tar_files()] expects a unevaluated symbol for the `name` argument
#'   and an unevaluated expression for `command`, whereas
#'   [tar_files_raw()] expects a character string for the `name` argument
#'   and an evaluated expression object for `command`. See the examples
#'   for a demo.
#' @details `tar_files()` creates a pair of targets, one upstream
#'   and one downstream.
#'
#'   The upstream target runs the `command` given
#'   by the `command` argument, and it should return a character vector
#'   of file paths. This upstream target needs to run on every
#'   [targets::tar_make()] because it needs to recheck which files
#'   are generated on disk. If your files are input files
#'   (not generated by the pipeline itself)
#'   and you do not want want to rerun the upstream target every pipeline,
#'   use [tar_files_input()] instead.
#'
#'   The downstream target is a dynamic branching target
#'   that applies `format = "file"`
#'   (or `format = "url"`) to track changes in the files.
#'   (URLs are input-only, they must already exist beforehand.)
#'
#'   This approach correctly dynamically
#'   branches over individual files. It makes sure any downstream dynamic
#'   branches only rerun *some* of their branches if the files/urls change.
#'   For more information, visit
#'   <https://github.com/ropensci/targets/issues/136> and
#'   <https://github.com/ropensci/drake/issues/1302>.
#' @return A list of two targets, one upstream and one downstream.
#'   The upstream one does some work and returns some file paths,
#'   and the downstream target is a pattern that applies `format = "file"`
#'   or `format = "url"`.
#'   See the "Target definition objects" section for background.
#' @inheritSection tar_map Target definition objects
#' @inheritParams targets::tar_target
#' @param name Name of the target.
#'   [tar_files()] expects a unevaluated symbol for the `name` argument
#'   and an unevaluated expression for `command`, whereas
#'   [tar_files_raw()] expects a character string for the `name` argument
#'   and an evaluated expression object for `command`. See the examples
#'   for a demo.
#' @param command R command for the target.
#'   [tar_files()] expects a unevaluated symbol for the `name` argument
#'   and an unevaluated expression for `command`, whereas
#'   [tar_files_raw()] expects a character string for the `name` argument
#'   and an evaluated expression object for `command`. See the examples
#'   for a demo.
#' @param format Character of length 1.
#'   Must be `"file"`, `"url"`, or `"aws_file"`. See the `format`
#'   argument of `targets::tar_target()` for details.
#' @param cue An optional object from `tar_cue()`
#'   to customize the rules that decide whether the target is up to date.
#'   Only applies to the downstream target. The upstream target always runs.
#' @examples
#' if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
#' targets::tar_dir({ # tar_dir() runs code from a temporary directory.
#' targets::tar_script({
#'   library(tarchetypes)
#'   # Do not use temp files in real projects
#'   # or else your targets will always rerun.
#'   paths <- unlist(replicate(2, tempfile()))
#'   file.create(paths)
#'   list(
#'     tar_files(name = x, command = paths),
#'     tar_files_raw(name = "y", command = quote(paths))
#'   )
#' })
#' targets::tar_make()
#' targets::tar_read(x)
#' })
#' }
tar_files <- function(
  name,
  command,
  tidy_eval = targets::tar_option_get("tidy_eval"),
  packages = targets::tar_option_get("packages"),
  library = targets::tar_option_get("library"),
  format = c("file", "file_fast", "url", "aws_file"),
  repository = targets::tar_option_get("repository"),
  iteration = targets::tar_option_get("iteration"),
  error = targets::tar_option_get("error"),
  memory = targets::tar_option_get("memory"),
  garbage_collection = targets::tar_option_get("garbage_collection"),
  deployment = targets::tar_option_get("deployment"),
  priority = targets::tar_option_get("priority"),
  resources = targets::tar_option_get("resources"),
  storage = targets::tar_option_get("storage"),
  retrieval = targets::tar_option_get("retrieval"),
  cue = targets::tar_option_get("cue"),
  description = targets::tar_option_get("description")
) {
  name <- targets::tar_deparse_language(substitute(name))
  envir <- tar_option_get("envir")
  command <- targets::tar_tidy_eval(substitute(command), envir, tidy_eval)
  format <- match.arg(format)
  tar_files_raw(
    name = name,
    command = command,
    packages = packages,
    library = library,
    format = format,
    repository = repository,
    iteration = iteration,
    error = error,
    memory = memory,
    garbage_collection = garbage_collection,
    deployment = deployment,
    priority = priority,
    resources = resources,
    storage = storage,
    retrieval = retrieval,
    cue = cue,
    description = description
  )
}
