#' Plot DBE vs ppm with Option for Interactive Plot
#'
#' This function generates a scatter plot of DBE (Double Bond Equivalent) versus parts per million (ppm) from the provided data.
#' It also provides the option to customize the appearance and to return an interactive `plotly` plot.

#' @inheritParams main_docu
#' @param df A data frame containing the data. The columns `ppm` (ppm values), `dbe` (DBE values),
#'           and `file_id` (for coloring the points) should be present in the data.
#'
#' @return A `ggplot` object or a `plotly` object depending on the `plotly` argument.
# @importFrom ggplot2 ggplot geom_point scale_color_gradientn labs theme_minimal theme annotation_custom
#' @import ggplot2
#' @importFrom plotly ggplotly
#' @import data.table
#'
# @examples
# uplot_dbe_vs_ppm(df = mf_data_demo, size_dots = 0.8, plotly = TRUE)

uplot_dbe_vs_ppm <- function(df,
                               size_dots = 0.5,
                               cex.axis = 1,  # Size of axis text
                               cex.lab = 1.4, # Size of axis label
                               plotly = FALSE,  # Option to return plotly object
                               ...
)
{
  # Check if required columns are present
  if (!all(c("ppm", "dbe", "file_id") %in% names(df))) {
    stop("Error: 'ppm', 'dbe', or 'file_id' column not found in the data frame.")
  }

  # Create the ggplot object
  p <- ggplot(df, aes(x = ppm, y = dbe, color = as.factor(file_id))) +
    geom_point(size = size_dots, alpha = 0.8) +
    labs(x = "ppm", y = "DBE", title = "DBE vs ppm") +
    scale_color_discrete(name = "File ID") +
    theme_minimal() +
    theme(
      axis.text = element_text(size = cex.axis),
      axis.title = element_text(size = cex.lab),
      plot.title = element_text(size = cex.lab + 1),
      legend.position = "right"
    )

  # Convert to plotly if requested
  if (plotly) {
    return(plotly::ggplotly(p))
  } else {
    return(p)
  }
}
