#' Kendrick Mass Defect (KMD) vs. Nominal Mass Plot
#'
#' @title Kendrick Mass Defect (KMD) vs. Nominal Mass Plot
#' @name uplot_kmd
#' @family plots
#'
#' @description
#' This function generates a scatter plot of Kendrick Mass Defect (KMD) versus
#' nominal mass (`nm`), with color-coding based on a specified variable
#' (`z_var`). Optionally, the plot can be returned as an interactive Plotly
#' object.
#'
#' @inheritParams main_docu
#' @inheritDotParams f_colorz
#'
#' @param df A `data.table` or `data.frame` containing columns:
#'   - `nm`  (nominal mass)
#'   - `kmd` (Kendrick mass defect)
#'   - the color variable given in `z_var`.
#' @param z_var Character. Name of the column used for color mapping.
#' @param palname Character. Palette name passed to `f_colorz()`.
#' @param size_dots Numeric. Point size.
#' @param col_bar Logical. (Reserved for future use; currently ignored.)
#' @param tf Logical. (Reserved for future use; currently passed to `f_colorz()` via `...` if desired.)
#' @param cex.axis Numeric. Axis text size.
#' @param cex.lab Numeric. Axis label size.
#'
#' @references
#' Kendrick E. (1963). A mass scale based on CH\eqn{_2} = 14.0000 for high
#' resolution mass spectrometry of organic compounds.
#' *Analytical Chemistry*, **35**, 2146–2154.
#'
#' Hughey C.A., Hendrickson C.L., Rodgers R.P., Marshall A.G., Qian K.N. (2001).
#' Kendrick mass defect spectrum: A compact visual analysis for ultrahigh-resolution
#' broadband mass spectra. *Analytical Chemistry*, **73**, 4676–4681.
#' \doi{10.1021/ac010560w}
#'
#' @import data.table ggplot2
#' @importFrom plotly ggplotly
#'
#' @return A `ggplot` object (or a `plotly` object if `plotly = TRUE`)
#'   showing KMD vs nominal mass.
#'
#' @examples
#' uplot_kmd(mf_data_demo, z_var = "norm_int", plotly = TRUE)
#'
#' @export
uplot_kmd <- function(df,
                      z_var   = "norm_int",
                      palname = "redblue",
                      size_dots = 0.5,
                      col_bar   = TRUE,
                      tf        = FALSE,
                      logo      = TRUE,
                      cex.axis  = 12,
                      cex.lab   = 15,
                      plotly    = FALSE,
                      ...) {

  zz <- color <- NULL

  # ensure data.table
  if (!data.table::is.data.table(df)) {
    df <- data.table::as.data.table(df)
  }

  # basic column checks
  if (!"nm"  %in% names(df)) stop("Missing required column: 'nm' (nominal mass).")
  if (!"kmd" %in% names(df)) stop("Missing required column: 'kmd' (Kendrick mass defect).")
  if (!z_var %in% names(df)) stop(paste("Missing required column:", z_var))

  # prepare data
  df_kmd_vs_nm <- df[, .(nm, kmd, zz = get(z_var))]
  data.table::setorder(df_kmd_vs_nm, zz)

  # create hex colors via f_colorz()
  df_kmd_vs_nm[, color := f_colorz(zz, palname = palname, ...)]

  # build ggplot; color is already hex, so use identity scale without legend
  p <- ggplot2::ggplot(df_kmd_vs_nm,
                       ggplot2::aes(x = nm, y = kmd, color = color)) +
    ggplot2::geom_point(size = size_dots) +
    ggplot2::scale_color_identity(guide = "none") +
    ggplot2::labs(
      x = "Nominal Mass (Da)",
      y = "Kendrick Mass Defect (KMD)",
      title = "Kendrick Mass Defect Plot"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(
      axis.text  = ggplot2::element_text(size = cex.axis),
      axis.title = ggplot2::element_text(size = cex.lab)
    )

  if (isTRUE(logo)) {
    p <- p + ggplot2::labs(caption = "UltraMassExplorer")
  }

  if (isTRUE(plotly)) {
    p <- plotly::ggplotly(p)
  }

  return(p)
}
