#' @title Plot Mass Spectrum
#' @name uplot_ms
#' @family plots
#' @description Plots the mass spectrum, showing magnitude versus mass-to-charge ratio (m/z).
#' Optionally reduces the data by selecting the top `data_reduction` most abundant peaks per spectrum.
#' @inheritParams main_docu
#' @param pl A data table that must contain columns for mass-to-charge ratio and peak magnitude (could be peak list or molecular formula data).
#' @param mass Character. Name of the column containing mass-to-charge or mass information (default = "mz").
#' @param peak_magnitude Character. Name of the column containing (relative) peak magnitude information (default = "i_magnitude").
#' @param label Character. Name of the column containing the names of the mass spectra to be displayed (default = "file_id").
#' @param data_reduction Numeric. The percentage of the most abundant peaks to select per spectrum.
#'   This value should be between 0 and 1 (default = 1, which means all data will be displayed).
#'   If set to 0, no data reduction will occur, but a minimum value of 0.01 will be used to ensure some data is displayed.
#' @import data.table
# @importFrom ggplot2 scale_color_identity scale_color_viridis_d geom_hline ggplot geom_bar geom_histogram coord_cartesian geom_point scale_color_gradientn labs theme_minimal theme annotation_custom
#' @import ggplot2
#' @importFrom plotly ggplotly
#'
#' @return A ggplot (class "ggplot") or plotly (class "htmlwidget") object
#' representing the mass spectrum.
#' @examples
#' uplot_ms(pl = peaklist_demo, data_reduction = 0.1, plotly = TRUE)
#' uplot_ms(pl = peaklist_demo, data_reduction = 1, plotly = FALSE)
#' @export

uplot_ms <- function(pl,
                     mass = "mz",
                     peak_magnitude = "i_magnitude",
                     label = "file_id",
                     logo = FALSE,
                     plotly = TRUE,
                     data_reduction = 1,
                     ...) {

  # Column checks
  if (!peak_magnitude %in% names(pl))
    stop("Data table does not have a column named ", peak_magnitude)
  if (!label %in% names(pl))
    stop("Data table does not have a column named ", label)

  # data_reduction validity
  if (!is.numeric(data_reduction) || data_reduction < 0 || data_reduction > 1)
    stop("`data_reduction` must be a numeric value between 0 and 1.")

  if (data_reduction == 0)
    data_reduction <- 0.01  # ensure some data is shown

  # Data reduction (top peaks per file)
  if (data_reduction < 1) {
    pl <- pl[, {
      .SD[order(-get(peak_magnitude))][1:ceiling(data_reduction * .N)]
    }, by = label]
  }

  pl[, (label) := as.factor(get(label))]

  # ---- ggplot ----
  gg <- ggplot(pl, aes(x = !!sym(mass), y = !!sym(peak_magnitude), color = !!sym(label))) +
    geom_segment(
      aes(xend = !!sym(mass), y = 0, yend = !!sym(peak_magnitude)),
      linewidth = 0.5
    ) +
    scale_color_viridis_d() +
    labs(
      title = "Mass Spectrum",
      x = paste0("m/z (", mass, ")"),
      y = paste0("Peak Magnitude (", peak_magnitude, ")")
    ) +
    theme_minimal(base_size = 14) +
    theme(legend.title = element_blank())

  if (logo) {
    gg <- gg + labs(caption = "UltraMassExplorer")
  }

  if (plotly) {
    return(plotly::ggplotly(gg))
  }

  return(gg)
}

