#' @title Utility Functions for Value-Price Analysis
#' @name utils
#' @description Internal utility functions used across the package.
NULL

#' Calculate Mode Using Kernel Density Estimation
#'
#' Estimates the mode of a numeric vector using kernel density estimation.
#' Falls back to median if density estimation fails.
#'
#' @param x A numeric vector.
#'
#' @return A single numeric value representing the estimated mode.
#'
#' @examples
#' set.seed(123)
#' x <- rnorm(100, mean = 5, sd = 1)
#' calculate_mode(x)
#'
#' @export
calculate_mode <- function(x) {


    x <- as.numeric(x)
    x <- x[is.finite(x)]

    if (length(x) == 0L) {
        return(NA_real_)
    }

    if (length(x) == 1L) {
        return(x[1L])
    }

    result <- tryCatch(
        {
            d <- stats::density(x)
            d$x[which.max(d$y)]
        },
        error = function(e) {
            stats::median(x, na.rm = TRUE)
        }
    )

    return(result)
}


#' Safe Percentage Calculation
#'
#' Calculates percentage while handling division by zero and non-finite values.
#'
#' @param numerator Numeric vector for the numerator.
#' @param denominator Numeric vector for the denominator.
#'
#' @return Numeric vector of percentages, with NA for invalid calculations.
#'
#' @examples
#' safe_pct(c(10, 20, 30), c(100, 0, 200))
#'
#' @export
safe_pct <- function(numerator, denominator) {

    numerator <- as.numeric(numerator)
    denominator <- as.numeric(denominator)

    result <- ifelse(
        is.finite(denominator) & denominator != 0,
        100 * numerator / denominator,
        NA_real_
    )

    return(as.numeric(result))
}


#' Safe RMSE Calculation
#'
#' Calculates Root Mean Squared Error handling non-finite values.
#'
#' @param actual Numeric vector of actual values.
#' @param predicted Numeric vector of predicted values.
#'
#' @return A single numeric value for RMSE, or NA if calculation not possible.
#'
#' @examples
#' actual <- c(1, 2, 3, 4, 5)
#' predicted <- c(1.1, 2.2, 2.9, 4.1, 5.2)
#' safe_rmse(actual, predicted)
#'
#' @export
safe_rmse <- function(actual, predicted) {

    valid <- is.finite(actual) & is.finite(predicted)

    if (!any(valid)) {
        return(NA_real_)
    }

    sqrt(mean((actual[valid] - predicted[valid])^2))
}


#' Safe MAE Calculation
#'
#' Calculates Mean Absolute Error handling non-finite values.
#'
#' @param actual Numeric vector of actual values.
#' @param predicted Numeric vector of predicted values.
#'
#' @return A single numeric value for MAE, or NA if calculation not possible.
#'
#' @examples
#' actual <- c(1, 2, 3, 4, 5)
#' predicted <- c(1.1, 2.2, 2.9, 4.1, 5.2)
#' safe_mae(actual, predicted)
#'
#' @export
safe_mae <- function(actual, predicted) {

    valid <- is.finite(actual) & is.finite(predicted)

    if (!any(valid)) {
        return(NA_real_)
    }

    mean(abs(actual[valid] - predicted[valid]))
}


#' Robust Summary Statistics with Bootstrap Confidence Intervals
#'
#' Computes mean, standard deviation, median, MAD, trimmed mean, and
#' bootstrap confidence intervals for a numeric vector.
#'
#' @param x Numeric vector.
#' @param bootstrap_reps Number of bootstrap replications for CI. Default 200.
#' @param trim_proportion Proportion to trim for trimmed mean. Default 0.10.
#'
#' @return A list containing:
#' \describe{
#'   \item{mean}{Arithmetic mean}
#'   \item{sd}{Standard deviation}
#'   \item{median}{Median}
#'   \item{mad}{Median Absolute Deviation (scaled)}
#'   \item{tmean}{Trimmed mean}
#'   \item{ci}{Vector of length 2 with 95 percent bootstrap CI bounds}
#' }
#'
#' @examples
#' set.seed(123)
#' x <- rnorm(50)
#' robust_summary(x)
#'
#' @export
robust_summary <- function(x, bootstrap_reps = 200L, trim_proportion = 0.10) {

    x <- x[is.finite(x)]

    if (length(x) == 0L) {
        return(list(
            mean = NA_real_,
            sd = NA_real_,
            median = NA_real_,
            mad = NA_real_,
            tmean = NA_real_,
            ci = c(NA_real_, NA_real_)
        ))
    }

    m <- mean(x)
    s <- stats::sd(x)
    md <- stats::median(x)
    tm <- mean(x, trim = trim_proportion)
    mad_val <- stats::mad(x, constant = 1.4826)

    if (length(x) >= 5L && bootstrap_reps > 20L) {
        boot_means <- replicate(bootstrap_reps, mean(sample(x, replace = TRUE)))
        ci <- stats::quantile(boot_means, c(0.025, 0.975), na.rm = TRUE, names = FALSE)
    } else {
        ci <- c(NA_real_, NA_real_)
    }

    list(
        mean = m,
        sd = s,
        median = md,
        mad = mad_val,
        tmean = tm,
        ci = ci
    )
}


#' Evaluate In-Sample Model Performance
#'
#' Computes various error metrics comparing predictions to actual values.
#'
#' @param predicted Numeric vector of predicted values (log scale).
#' @param actual Numeric vector of actual values (log scale).
#'
#' @return A list containing:
#' \describe{
#'   \item{mae_log}{MAE in log scale}
#'   \item{rmse_log}{RMSE in log scale}
#'   \item{mae_orig}{MAE in original scale (after exp transformation)}
#'   \item{rmse_orig}{RMSE in original scale}
#'   \item{mae_rel_range}{MAE as percentage of range}
#' }
#'
#' @examples
#' set.seed(123)
#' actual <- log(runif(50, 100, 200))
#' predicted <- actual + rnorm(50, 0, 0.1)
#' evaluate_insample(predicted, actual)
#'
#' @export
evaluate_insample <- function(predicted, actual) {

    valid <- is.finite(predicted) & is.finite(actual)

    if (!any(valid)) {
        return(list(
            mae_log = NA_real_,
            rmse_log = NA_real_,
            mae_orig = NA_real_,
            rmse_orig = NA_real_,
            mae_rel_range = NA_real_
        ))
    }

    p <- predicted[valid]
    a <- actual[valid]

    mae_log <- mean(abs(a - p))
    rmse_log <- sqrt(mean((a - p)^2))

    mae_orig <- mean(abs(exp(a) - exp(p)))
    rmse_orig <- sqrt(mean((exp(a) - exp(p))^2))

    rng <- diff(range(a))
    mae_rel_range <- if (rng > 0) mae_log / rng * 100 else NA_real_

    list(
        mae_log = mae_log,
        rmse_log = rmse_log,
        mae_orig = mae_orig,
        rmse_orig = rmse_orig,
        mae_rel_range = mae_rel_range
    )
}


#' Compute In-Sample R-squared
#'
#' Calculates the coefficient of determination (R-squared) for predictions.
#'
#' @param actual Numeric vector of actual values.
#' @param predicted Numeric vector of predicted values.
#'
#' @return A single numeric R-squared value, or NA if not computable.
#'
#' @examples
#' actual <- c(1, 2, 3, 4, 5)
#' predicted <- c(1.1, 1.9, 3.1, 4.0, 4.9)
#' compute_r2(actual, predicted)
#'
#' @export
compute_r2 <- function(actual, predicted) {

    valid <- is.finite(actual) & is.finite(predicted)

    if (!any(valid)) {
        return(NA_real_)
    }

    a <- actual[valid]
    p <- predicted[valid]

    ss_res <- sum((a - p)^2)
    ss_tot <- sum((a - mean(a))^2)

    if (ss_tot <= 0) {
        return(NA_real_)
    }

    1 - ss_res / ss_tot
}


#' Extract R-squared Values from rstanarm Model
#'
#' Attempts to extract R-squared using loo_R2, falling back to bayes_R2.
#'
#' @param fit A fitted rstanarm model object.
#' @param verbose Logical. Print messages about extraction method. Default FALSE.
#'
#' @return A named numeric vector with mean, median, and mode R-squared values.
#'
#' @examples
#' \donttest{
#' if (requireNamespace("rstanarm", quietly = TRUE)) {
#'   data(mtcars)
#'   fit <- rstanarm::stan_glm(mpg ~ wt, data = mtcars,
#'                             chains = 2, iter = 1000, refresh = 0)
#'   get_r2_values(fit)
#' }
#' }
#'
#' @export
get_r2_values <- function(fit, verbose = FALSE) {

    if (!requireNamespace("rstanarm", quietly = TRUE)) {
        stop("Package 'rstanarm' is required but not installed.")
    }

    if (!requireNamespace("loo", quietly = TRUE)) {
        stop("Package 'loo' is required but not installed.")
    }

    r2_loo_num <- suppressWarnings(tryCatch(
        {
          r2_loo_obj <- rstanarm::loo_R2(fit)

            if (is.numeric(r2_loo_obj) && length(r2_loo_obj) == 1L) {
                as.numeric(r2_loo_obj)
            } else if (is.list(r2_loo_obj) && !is.null(r2_loo_obj$R2)) {
                as.numeric(r2_loo_obj$R2)
            } else if (is.matrix(r2_loo_obj) && length(r2_loo_obj) >= 1L) {
                as.numeric(r2_loo_obj[1L])
            } else {
                NA_real_
            }
        },
        error = function(e) NA_real_
    ))

    if (!is.finite(r2_loo_num)) {

        if (verbose) {
            message("loo_R2 failed or not finite; using bayes_R2 draws.")
        }

        r2_draws <- drop(rstanarm::bayes_R2(fit))
        r2_draws <- as.numeric(r2_draws)

        c(
            mean = mean(r2_draws, na.rm = TRUE),
            median = stats::median(r2_draws, na.rm = TRUE),
            mode = calculate_mode(r2_draws)
        )
    } else {
        c(mean = r2_loo_num, median = r2_loo_num, mode = r2_loo_num)
    }
}


#' Check if Required Package is Available
#'
#' Checks for package availability and provides informative error message.
#'
#' @param pkg Character string with package name.
#' @param reason Character string explaining why the package is needed.
#'
#' @return TRUE invisibly if package is available, otherwise stops with error.
#'
#' @examples
#' check_package("stats", "basic statistics")
#'
#' @export
check_package <- function(pkg, reason = "this functionality") {

    if (!requireNamespace(pkg, quietly = TRUE)) {
        stop(
            sprintf(
                "Package '%s' is required for %s but is not installed.",
                pkg,
                reason
            ),
            call. = FALSE
        )
    }

    invisible(TRUE)
}
