#' Coerce to predxhaz object
#'
#' @description Normalize output to a standard format like predxahz with three components:
#'
#' @param x object to be coerced
#' @param ... further arguments passed to specific methods
#' @return list(time=, survival=, hazard=) with class "predxhaz"
#'
#' @export
#'
as.predxhaz <- function(x, ...) UseMethod("as.predxhaz")

#' @export
#'
as.predxhaz.default <- function(x, ...) {
  if (is.list(x) && all(c("time", "survival", "hazard") %in% names(x))) {
    class(x) <- unique(c("predxhaz", class(x)))
    return(x)
  }

  if (is.list(x) && length(x) >= 1 && !is.null(x[[1]]$times.pts)) {
    time <- vapply(x, function(e) unique(e$times.pts), numeric(1))
    survival <- vapply(x, function(e) mean(e$survival, na.rm = TRUE), numeric(1))
    hazard <- vapply(x, function(e) {
      w <- e$survival
      num <- sum(e$hazard * w, na.rm = TRUE)
      den <- sum(w, na.rm = TRUE)
      if (!is.finite(num) || den == 0) mean(e$hazard, na.rm = TRUE) else num/den
    }, numeric(1))
    out <- list(time = time, survival = survival, hazard = hazard)
    class(out) <- c("predxhaz","list")
    attr(out, "baseline") <- attr(x, "baseline", exact = TRUE)
    return(out)
  }
  stop("Don't know how to coerce to 'predxhaz'.", call. = FALSE)
}

#' @export
#' @method as.predxhaz predMexhaz
as.predxhaz.predMexhaz <- function(x, ...) {
  if (!is.null(x$results)) {
    out <- list(time = x$results$time.pts,
                survival = x$results$surv,
                hazard   = x$results$hazard)
    class(out) <- c("predxhaz","list")
    attr(out, "baseline") <- attr(x, "baseline", exact = TRUE)
    return(out)
  }
  as.predxhaz.default(x, ...)
}
