#' @title Predictions of excess hazard and net Survival from an \code{constant}
#' object
#'
#' @description Function to predict excess hazard and net survival based on an
#' object of class \code{constant}. The function allows the
#' predictions at several time points but not exceeding the maximum time of
#' follow-up from the baseline model.
#'
#'
#' @param object An object of class constant
#'
#' @param new.data new.data where is covariates
#'
#' @param times.pts time in year scale to calculate the excess hazard. The
#' default value is NULL. In this case, time variable must be provided in the
#' new.data
#'
#' @param baseline default is survival baseline; put \code{baseline = FALSE}
#'  to estimate the net survival with covariates
#'
#' @param ... additional arguments affecting the predictions of excess hazard
#' and net survival
#'
#' @keywords predict.constant
#'
#' @return An object of class predxhaz. The return of this fonction can be
#' used to produce graphics of excess hazard or net survival, when times.pts argument is provided. This object
#' contains:
#'
#'
#' \item{times.pts}{the times value in year at which the excess hazard and or the net survival have been estimated}
#'
#' \item{hazard}{the excess hazard values based on the model of interest}
#'
#' \item{survival}{the net survival values based on the model of interest}
#'
#'
#'
#' @author Juste Goungounga, Robert Darlin Mba, Nathalie Graff\'eo and Roch Giorgi
#'
#' @references Goungounga JA, Touraine C, Graff\'eo N, Giorgi R;
#' CENSUR working survival group. Correcting for misclassification
#' and selection effects in estimating net survival in clinical trials.
#' BMC Med Res Methodol. 2019 May 16;19(1):104.
#' doi: 10.1186/s12874-019-0747-3. PMID: 31096911; PMCID: PMC6524224.
#' (\href{https://pubmed.ncbi.nlm.nih.gov/31096911/}{PubMed})
#'
#' Touraine C, Graff\'eo N, Giorgi R; CENSUR working survival group.
#' More accurate cancer-related excess mortality through correcting
#' background mortality for extra variables.
#' Stat Methods Med Res. 2020 Jan;29(1):122-136.
#' doi: 10.1177/0962280218823234. Epub 2019 Jan 23. PMID: 30674229.
#' (\href{https://pubmed.ncbi.nlm.nih.gov/30674229/}{PubMed})
#'
#' Mba RD, Goungounga JA, Graff\'eo N, Giorgi R; CENSUR working survival group.
#' Correcting inaccurate background mortality in excess hazard models
#' through breakpoints. BMC Med Res Methodol. 2020 Oct 29;20(1):268.
#' doi: 10.1186/s12874-020-01139-z. PMID: 33121436; PMCID: PMC7596976.
#' (\href{https://pubmed.ncbi.nlm.nih.gov/33121436/}{PubMed})
#'
#'
#'
#' @seealso \code{\link{xhaz}}, \code{\link{print.bsplines}}, \code{\link{print.constant}}
#'
#' @examples
#'
#' # load the data set in the package
#' library("xhaz")
#' library("numDeriv")
#'
#' # load the data sets 'simuData
#'
#' data("simuData", package = "xhaz")
#'
#' #define the levels of variable sex
#' levels(simuData$sex) <- c("male", "female")
#'
#' # Esteve et al. model
#'
#' set.seed(1980)
#' simuData2 <- simuData[sample(nrow(simuData), size = 500), ]
#'
#' fit.estv2 <- xhaz(formula = Surv(time_year, status) ~ agec + race,
#'                   data = simuData2,
#'                   ratetable = survexp.us,
#'                   interval = c(0, NA, NA, NA, NA, NA, 6),
#'                   rmap = list(age = 'age', sex = 'sex', year = 'date'),
#'                   baseline = "constant", pophaz = "classic")
#'
#'
#' predict_est <- predict(object = fit.estv2,
#'                        new.data = simuData2,
#'                        times.pts = c(seq(0, 4, 1)),
#'                        baseline = TRUE)
#' predict_est
#'
#' @export

predict.constant <- function(object,
                             new.data = NULL,
                             times.pts = NULL,
                             baseline = TRUE,
                             ...) {
    Call <- match.call()
    int  <- object$interval
    stopifnot(inherits(object, "constant"))

    alpha_idx     <- grep("alpha", names(object$coefficients), fixed = TRUE)
    length_alpha  <- length(alpha_idx)
    n_segments    <- length(int) - 1
    total_coef    <- length(object$coefficients)
    nvar          <- total_coef - n_segments - length_alpha
    coeffPred     <- object$coefficients[seq_len(nvar)]
    log_haz_seg   <- object$coefficients[(nvar + 1):(nvar + n_segments)]
    coefPWSE      <- exp(log_haz_seg)

    m <- eval(object$terms, sys.parent())
    time_name  <- toString(as.name(m[[2]][[2]]))
    event_name <- toString(as.name(m[[2]][[3]]))

    if (is.null(new.data)) new.data <- object$data
    if (is.null(times.pts)) {
      times.pts <- sort(unique(new.data[[time_name]]))
    } else {
      times.pts <- sort(unique(as.numeric(times.pts)))
    }

    if (!(event_name %in% names(new.data))) new.data[[event_name]] <- 0
    if (!(time_name  %in% names(new.data))) new.data[[time_name]]  <- times.pts[1]

    xx <- as.data.frame(model.matrix(object$terms, new.data))[, -1, drop = FALSE]

    if (max(times.pts) > max(int))
      stop("times.pts must be \u2264 max(object$interval).")

    haz0 <- vapply(times.pts, function(t) {
      idx <- findInterval(t, int, rightmost.closed = TRUE, all.inside = TRUE)
      idx <- max(1, min(idx, n_segments))
      coefPWSE[idx]
    }, numeric(1))

    Cum0 <- vapply(times.pts, function(t) {
      seg <- findInterval(t, int, rightmost.closed = TRUE, all.inside = TRUE)
      seg <- max(1, min(seg, n_segments))
      full <- if (seg > 1) sum(coefPWSE[1:(seg-1)] * diff(int)[1:(seg-1)]) else 0
      frac <- coefPWSE[seg] * (t - int[seg])
      full + frac
    }, numeric(1))
    S0 <- exp(-Cum0)

    if (isTRUE(baseline)) {
      haz <- haz0
      S   <- S0
    } else {
      rrBetaZ <- as.numeric(exp(as.matrix(xx) %*% coeffPred))
      haz <- haz0 * mean(rrBetaZ)
      S   <- vapply(S0, function(s0) mean(s0 ^ rrBetaZ), numeric(1))
    }

    new_predxhaz(
      time     = times.pts,
      hazard   = haz,
      survival = S,
      baseline = "constant",
      attrs = list(
        call         = Call,
        pophaz       = object$pophaz,
        interval     = object$interval,
        coefficients = object$coefficients,
        source_class = "xhaz.constant"
      )
    )

}
