use super::*;
use extendr_ffi::{R_BlankString, R_NaString, R_NilValue, Rf_xlength, R_CHAR, SEXPTYPE, TYPEOF};












#[derive(Clone)]
pub struct Rstr {
    pub(crate) robj: Robj,
}




pub(crate) unsafe fn charsxp_to_str(charsxp: SEXP) -> Option<&'static str> {
    assert_eq!(TYPEOF(charsxp), SEXPTYPE::CHARSXP);
    if charsxp == R_NilValue {
        None
    } else if charsxp == R_NaString {
        Some(<&str>::na())
    } else if charsxp == R_BlankString {
        Some("")
    } else {
        let length = Rf_xlength(charsxp);
        let all_bytes =
            std::slice::from_raw_parts(R_CHAR(charsxp).cast(), length.try_into().unwrap());
        Some(std::str::from_utf8_unchecked(all_bytes))
    }
}

impl Rstr {
    
    pub fn from_string(val: &str) -> Self {
        Rstr {
            robj: Robj::from_sexp(str_to_character(val)),
        }
    }

    
    
    pub fn as_str(&self) -> &str {
        self.into()
    }
}

impl AsRef<str> for Rstr {
    
    fn as_ref(&self) -> &str {
        self.as_str()
    }
}

impl From<String> for Rstr {
    
    fn from(s: String) -> Self {
        Rstr::from_string(&s)
    }
}

impl From<&str> for Rstr {
    
    fn from(s: &str) -> Self {
        Rstr::from_string(s)
    }
}

impl From<&Rstr> for &str {
    fn from(value: &Rstr) -> Self {
        unsafe {
            let charsxp = value.robj.get();
            rstr::charsxp_to_str(charsxp).unwrap()
        }
    }
}

impl From<Option<String>> for Rstr {
    fn from(value: Option<String>) -> Self {
        if let Some(string) = value {
            Self::from(string)
        } else {
            Self { robj: na_string() }
        }
    }
}

impl Deref for Rstr {
    type Target = str;

    
    fn deref(&self) -> &Self::Target {
        self.as_str()
    }
}


impl PartialEq<Rstr> for Rstr {
    fn eq(&self, other: &Rstr) -> bool {
        unsafe { self.robj.get() == other.robj.get() }
    }
}



impl PartialEq<str> for Rstr {
    
    fn eq(&self, other: &str) -> bool {
        self.as_str() == other
    }
}

impl PartialEq<Rstr> for &str {
    
    fn eq(&self, other: &Rstr) -> bool {
        *self == other.as_str()
    }
}

impl PartialEq<&str> for Rstr {
    
    fn eq(&self, other: &&str) -> bool {
        self.as_str() == *other
    }
}

impl PartialEq<Rstr> for &&str {
    
    fn eq(&self, other: &Rstr) -> bool {
        **self == other.as_str()
    }
}

impl std::fmt::Debug for Rstr {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.is_na() {
            write!(f, "NA_CHARACTER")
        } else {
            let s = self.as_str();
            write!(f, "{:?}", s)
        }
    }
}

impl std::fmt::Display for Rstr {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let s = self.as_str();
        write!(f, "{}", s)
    }
}

impl CanBeNA for Rstr {
    fn is_na(&self) -> bool {
        unsafe { self.robj.get() == R_NaString }
    }

    fn na() -> Self {
        unsafe {
            Self {
                robj: Robj::from_sexp(R_NaString),
            }
        }
    }
}
