



#![allow(clippy::match_same_arms)] // https://github.com/rust-lang/rust-clippy/issues/12044

#[path = "version.rs"]
mod version;
use self::version::{Version, rustc_version};

#[path = "src/gen/build.rs"]
mod generated;

use std::{env, str};

fn main() {
    println!("cargo:rerun-if-changed=build.rs");
    println!("cargo:rerun-if-changed=src/gen/build.rs");
    println!("cargo:rerun-if-changed=version.rs");

    #[cfg(feature = "unsafe-assume-single-core")]
    println!("cargo:rustc-cfg=portable_atomic_unsafe_assume_single_core");
    #[cfg(feature = "s-mode")]
    println!("cargo:rustc-cfg=portable_atomic_s_mode");
    #[cfg(feature = "force-amo")]
    println!("cargo:rustc-cfg=portable_atomic_force_amo");
    #[cfg(feature = "disable-fiq")]
    println!("cargo:rustc-cfg=portable_atomic_disable_fiq");

    let target = &*env::var("TARGET").expect("TARGET not set");
    let target_arch = &*env::var("CARGO_CFG_TARGET_ARCH").expect("CARGO_CFG_TARGET_ARCH not set");
    let target_os = &*env::var("CARGO_CFG_TARGET_OS").expect("CARGO_CFG_TARGET_OS not set");

    let version = match rustc_version() {
        Some(version) => version,
        None => {
            if env::var_os("PORTABLE_ATOMIC_DENY_WARNINGS").is_some() {
                panic!("unable to determine rustc version")
            }
            println!(
                "cargo:warning={}: unable to determine rustc version; assuming latest stable rustc (1.{})",
                env!("CARGO_PKG_NAME"),
                Version::LATEST.minor
            );
            Version::LATEST
        }
    };

    if version.minor >= 80 {
        println!(
            r#"cargo:rustc-check-cfg=cfg(target_feature,values("zacas","fast-serialization","load-store-on-cond","distinct-ops","miscellaneous-extensions-3"))"#
        );

        
        
        println!(
            "cargo:rustc-check-cfg=cfg(portable_atomic_disable_fiq,portable_atomic_force_amo,portable_atomic_ll_sc_rmw,portable_atomic_new_atomic_intrinsics,portable_atomic_no_asm,portable_atomic_no_asm_maybe_uninit,portable_atomic_no_atomic_64,portable_atomic_no_atomic_cas,portable_atomic_no_atomic_load_store,portable_atomic_no_atomic_min_max,portable_atomic_no_cfg_target_has_atomic,portable_atomic_no_cmpxchg16b_intrinsic,portable_atomic_no_cmpxchg16b_target_feature,portable_atomic_no_const_mut_refs,portable_atomic_no_const_raw_ptr_deref,portable_atomic_no_const_transmute,portable_atomic_no_core_unwind_safe,portable_atomic_no_diagnostic_namespace,portable_atomic_no_offset_of,portable_atomic_no_strict_provenance,portable_atomic_no_stronger_failure_ordering,portable_atomic_no_track_caller,portable_atomic_no_unsafe_op_in_unsafe_fn,portable_atomic_pre_llvm_15,portable_atomic_pre_llvm_16,portable_atomic_pre_llvm_18,portable_atomic_pre_llvm_20,portable_atomic_s_mode,portable_atomic_sanitize_thread,portable_atomic_target_feature,portable_atomic_unsafe_assume_single_core,portable_atomic_unstable_asm,portable_atomic_unstable_asm_experimental_arch,portable_atomic_unstable_cfg_target_has_atomic,portable_atomic_unstable_isa_attribute)"
        );
        
        // grep -F 'target_feature_fallback("' build.rs | grep -Ev '^ *//' | sed -E 's/^.*target_feature_fallback\(//; s/",.*$/"/' | LC_ALL=C sort -u | tr '\n' ',' | sed -E 's/,$/\n/'
        println!(
            r#"cargo:rustc-check-cfg=cfg(portable_atomic_target_feature,values("cmpxchg16b","distinct-ops","fast-serialization","load-store-on-cond","lse","lse128","lse2","mclass","miscellaneous-extensions-3","quadword-atomics","rcpc3","v6","zaamo","zabha","zacas"))"#
        );
    }

    
    if !version.probe(79, 2024, 4, 10) {
        
        
        
        
        
        
        println!("cargo:rerun-if-env-changed=CARGO_ENCODED_RUSTFLAGS");
        println!("cargo:rerun-if-env-changed=RUSTFLAGS");
        println!("cargo:rerun-if-env-changed=CARGO_BUILD_RUSTFLAGS");
        let mut target_upper = target.replace(|c: char| c == '-' || c == '.', "_");
        target_upper.make_ascii_uppercase();
        println!("cargo:rerun-if-env-changed=CARGO_TARGET_{}_RUSTFLAGS", target_upper);
    }

    
    
    

    
    if !version.probe(45, 2020, 5, 29) {
        println!("cargo:rustc-cfg=portable_atomic_no_atomic_min_max");
    }
    
    if !version.probe(46, 2020, 7, 1) {
        println!("cargo:rustc-cfg=portable_atomic_no_track_caller");
    }
    
    if !version.probe(52, 2021, 3, 10) {
        println!("cargo:rustc-cfg=portable_atomic_no_unsafe_op_in_unsafe_fn");
    }
    
    if !version.probe(56, 2021, 7, 28) {
        println!("cargo:rustc-cfg=portable_atomic_no_const_transmute");
    }
    
    if !version.probe(56, 2021, 8, 1) {
        println!("cargo:rustc-cfg=portable_atomic_no_core_unwind_safe");
    }
    
    if !version.probe(58, 2021, 11, 14) {
        println!("cargo:rustc-cfg=portable_atomic_no_const_raw_ptr_deref");
    }
    
    if !version.probe(64, 2022, 7, 18) {
        println!("cargo:rustc-cfg=portable_atomic_no_stronger_failure_ordering");
    }
    
    if !version.probe(74, 2023, 8, 23) {
        println!("cargo:rustc-cfg=portable_atomic_no_asm_maybe_uninit");
    }
    
    if version.minor < 77 {
        println!("cargo:rustc-cfg=portable_atomic_no_offset_of");
    }
    // #[diagnostic] stabilized in Rust 1.78 (nightly-2024-03-09): https://github.com/rust-lang/rust/pull/119888
    if !version.probe(78, 2024, 3, 8) {
        println!("cargo:rustc-cfg=portable_atomic_no_diagnostic_namespace");
    }
    
    if !version.probe(83, 2024, 9, 15) {
        println!("cargo:rustc-cfg=portable_atomic_no_const_mut_refs");
    }
    
    if !version.probe(84, 2024, 10, 21) {
        println!("cargo:rustc-cfg=portable_atomic_no_strict_provenance");
    }

    
    let no_asm = !version.probe(59, 2021, 12, 15);
    if no_asm {
        if version.nightly
            && version.probe(46, 2020, 6, 20)
            && ((target_arch != "x86" && target_arch != "x86_64") || version.llvm >= 10)
            && is_allowed_feature("asm")
        {
            
            
            
            
            
            
            println!("cargo:rustc-cfg=portable_atomic_unstable_asm");
        }
        println!("cargo:rustc-cfg=portable_atomic_no_asm");
    } else {
        match target_arch {
            "arm64ec" | "s390x" => {
                
                if !version.probe(84, 2024, 11, 10) {
                    if version.nightly
                        && (target_arch != "s390x" || version.probe(71, 2023, 5, 8))
                        && is_allowed_feature("asm_experimental_arch")
                    {
                        
                        
                        
                        println!("cargo:rustc-cfg=portable_atomic_unstable_asm_experimental_arch");
                    } else {
                        println!("cargo:rustc-cfg=portable_atomic_no_asm");
                    }
                }
            }
            "powerpc64" => {
                
                if version.nightly
                    && version.probe(60, 2022, 2, 12)
                    && is_allowed_feature("asm_experimental_arch")
                {
                    println!("cargo:rustc-cfg=portable_atomic_unstable_asm_experimental_arch");
                }
            }
            _ => {}
        }
    }

    
    if !version.probe(60, 2022, 2, 10) {
        if version.nightly
            && version.probe(40, 2019, 10, 13)
            && is_allowed_feature("cfg_target_has_atomic")
        {
            
            
            println!("cargo:rustc-cfg=portable_atomic_unstable_cfg_target_has_atomic");
        } else {
            println!("cargo:rustc-cfg=portable_atomic_no_cfg_target_has_atomic");
            let target = &*convert_custom_linux_target(target);
            if generated::NO_ATOMIC_CAS.contains(&target) {
                println!("cargo:rustc-cfg=portable_atomic_no_atomic_cas");
            }
            if generated::NO_ATOMIC_64.contains(&target) {
                println!("cargo:rustc-cfg=portable_atomic_no_atomic_64");
            } else {
                
            }
        }
    }
    
    if generated::NO_ATOMIC.contains(&target) {
        println!("cargo:rustc-cfg=portable_atomic_no_atomic_load_store");
    }

    if version.llvm < 20 {
        println!("cargo:rustc-cfg=portable_atomic_pre_llvm_20");
        if version.llvm < 18 {
            println!("cargo:rustc-cfg=portable_atomic_pre_llvm_18");
            if version.llvm < 16 {
                println!("cargo:rustc-cfg=portable_atomic_pre_llvm_16");
                if version.llvm < 15 {
                    println!("cargo:rustc-cfg=portable_atomic_pre_llvm_15");
                }
            }
        }
    }

    if version.nightly {
        
        let sanitize = env::var("CARGO_CFG_SANITIZE").unwrap_or_default();
        if sanitize.contains("thread") {
            
            
            
            
            println!("cargo:rustc-cfg=portable_atomic_sanitize_thread");
        }
    }

    match target_arch {
        "x86_64" => {
            
            if !version.probe(69, 2023, 2, 28) {
                println!("cargo:rustc-cfg=portable_atomic_no_cmpxchg16b_target_feature");
            }
            
            
            if version.nightly && !version.probe(70, 2023, 3, 23) {
                println!("cargo:rustc-cfg=portable_atomic_no_cmpxchg16b_intrinsic");
            }

            
            if needs_target_feature_fallback(&version, Some(69)) {
                
                // https://github.com/rust-lang/rust/blob/1.68.0/compiler/rustc_target/src/spec/x86_64_apple_darwin.rs#L8
                // https://github.com/rust-lang/rust/blob/1.68.0/compiler/rustc_target/src/spec/apple_base.rs#L69-L70
                
                
                
                
                let is_apple = env::var("CARGO_CFG_TARGET_VENDOR").unwrap_or_default() == "apple";
                let cmpxchg16b = is_apple;
                
                
                target_feature_fallback("cmpxchg16b", cmpxchg16b);
            }
        }
        "aarch64" | "arm64ec" => {
            
            
            if version.nightly && version.probe(64, 2022, 6, 29) {
                println!("cargo:rustc-cfg=portable_atomic_new_atomic_intrinsics");
            }

            
            if !version.probe(82, 2024, 8, 29) || needs_target_feature_fallback(&version, None) {
                
                
                // https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/AArch64/AArch64Processors.td#L1230
                // https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/AArch64/AArch64Processors.td#L891
                
                
                
                let is_macos = target_os == "macos";
                let mut lse = is_macos;
                target_feature_fallback("lse2", is_macos);
                lse |= target_feature_fallback("lse128", false);
                target_feature_fallback("rcpc3", false);
                
                if needs_target_feature_fallback(&version, Some(61)) {
                    target_feature_fallback("lse", lse);
                }
            }

            
            
            let is_apple = env::var("CARGO_CFG_TARGET_VENDOR").unwrap_or_default() == "apple";
            if is_apple || target_cpu().map_or(false, |cpu| cpu.starts_with("apple-")) {
                println!("cargo:rustc-cfg=portable_atomic_ll_sc_rmw");
            }
        }
        "arm" => {
            
            
            if version.nightly && !version.probe(67, 2022, 11, 5) {
                println!("cargo:rustc-cfg=portable_atomic_unstable_isa_attribute");
            }

            if needs_target_feature_fallback(&version, None) {
                // #[cfg(target_feature = "v7")] and others don't work on stable.
                
                
                let mut subarch =
                    strip_prefix(target, "arm").or_else(|| strip_prefix(target, "thumb")).unwrap();
                subarch = strip_prefix(subarch, "eb").unwrap_or(subarch); 
                subarch = subarch.split('-').next().unwrap(); 
                subarch = subarch.split('.').next().unwrap(); 
                let mut known = true;
                
                let mut mclass = false;
                match subarch {
                    "v7" | "v7a" | "v7neon" | "v7s" | "v7k" | "v8" | "v8a" | "v9" | "v9a" => {} 
                    "v7r" | "v8r" | "v9r" => {} 
                    "v6m" | "v7em" | "v7m" | "v8m" => mclass = true,
                    
                    // https://github.com/rust-lang/rust/blob/1.84.0/compiler/rustc_target/src/spec/targets/arm_linux_androideabi.rs#L18
                    _ if target == "arm-linux-androideabi" => subarch = "v5te",
                    
                    // https://github.com/rust-lang/rust/blob/1.84.0/compiler/rustc_target/src/spec/targets/armeb_unknown_linux_gnueabi.rs#L18
                    _ if target == "armeb-unknown-linux-gnueabi" => subarch = "v8",
                    
                    "" => subarch = "v6",
                    "v4t" | "v5te" | "v6" | "v6k" => {}
                    _ => {
                        known = false;
                        if env::var_os("PORTABLE_ATOMIC_DENY_WARNINGS").is_some() {
                            panic!("unrecognized Arm subarch: {}", target)
                        }
                        println!(
                            "cargo:warning={}: unrecognized Arm subarch: {}",
                            env!("CARGO_PKG_NAME"),
                            target
                        );
                    }
                }
                let v6 = known
                    && (subarch.starts_with("v6")
                        || subarch.starts_with("v7")
                        || subarch.starts_with("v8")
                        || subarch.starts_with("v9"));
                target_feature_fallback("v6", v6);
                target_feature_fallback("mclass", mclass);
            }
        }
        "riscv32" | "riscv64" => {
            
            
            
            // https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/RISCV/RISCVFeatures.td#L233-L247
            
            // https://github.com/gcc-mirror/gcc/blob/08693e29ec186fd7941d0b73d4d466388971fe2f/gcc/config/riscv/arch-canonicalize#L45-L46
            
            let mut zaamo = false;
            
            // https://github.com/rust-lang/rust/blob/1.84.0/compiler/rustc_target/src/target_features.rs#L425
            
            
            
            zaamo |= target_feature_fallback("zacas", false);
            
            if !version.probe(83, 2024, 10, 1) || needs_target_feature_fallback(&version, None) {
                if version.llvm >= 19 {
                    
                    
                    zaamo |= target_feature_fallback("zabha", false);
                }
                
                target_feature_fallback("zaamo", zaamo);
            }
        }
        "powerpc64" => {
            
            if !version.probe(83, 2024, 9, 27) || needs_target_feature_fallback(&version, None) {
                let mut pwr8_features = false;
                if let Some(cpu) = target_cpu() {
                    if let Some(mut cpu_version) = strip_prefix(&cpu, "pwr") {
                        cpu_version = strip_suffix(cpu_version, "x").unwrap_or(cpu_version); 
                        if let Ok(cpu_version) = cpu_version.parse::<u32>() {
                            pwr8_features = cpu_version >= 8;
                        }
                    } else {
                        // https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/PowerPC/PPC.td#L714
                        // https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/PowerPC/PPC.td#L483
                        
                        
                        // https://github.com/llvm/llvm-project/blob/llvmorg-12.0.0/llvm/lib/Target/PowerPC/PPC.td#L370
                        pwr8_features = cpu == "future" || cpu == "ppc64le";
                    }
                } else {
                    // powerpc64le is pwr8 by default https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/PowerPC/PPC.td#L714
                    
                    pwr8_features = env::var("CARGO_CFG_TARGET_ENDIAN")
                        .expect("CARGO_CFG_TARGET_ENDIAN not set")
                        == "little";
                }
                // power8 features: https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/PowerPC/PPC.td#L409
                
                target_feature_fallback("quadword-atomics", pwr8_features);
            }
        }
        "s390x" => {
            let mut arch9_features = false; 
            let mut arch13_features = false; 
            if let Some(cpu) = target_cpu() {
                
                
                // https://github.com/gcc-mirror/gcc/blob/releases/gcc-14.2.0/gcc/config/s390/s390.opt#L58-L125
                if let Some(arch_version) = strip_prefix(&cpu, "arch") {
                    if let Ok(arch_version) = arch_version.parse::<u32>() {
                        arch9_features = arch_version >= 9;
                        arch13_features = arch_version >= 13;
                    }
                } else {
                    match &*cpu {
                        "z196" | "zEC12" | "z13" | "z14" => arch9_features = true,
                        "z15" | "z16" => {
                            arch9_features = true;
                            arch13_features = true;
                        }
                        _ => {}
                    }
                }
            }
            
            // https://github.com/rust-lang/rust/blob/1.84.0/compiler/rustc_target/src/target_features.rs#L547
            // arch9 features: https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/SystemZ/SystemZFeatures.td#L103
            
            target_feature_fallback("fast-serialization", arch9_features);
            
            target_feature_fallback("load-store-on-cond", arch9_features);
            
            target_feature_fallback("distinct-ops", arch9_features);
            // arch13 features: https://github.com/llvm/llvm-project/blob/llvmorg-20.1.0-rc1/llvm/lib/Target/SystemZ/SystemZFeatures.td#L301
            
            target_feature_fallback("miscellaneous-extensions-3", arch13_features);
        }
        _ => {}
    }
}




// - #[cfg(target_feature = "unstable_target_feature")] doesn't work on stable.





// [RFC2045]: https://rust-lang.github.io/rfcs/2045-target-feature.html#backend-compilation-options
fn needs_target_feature_fallback(version: &Version, stable: Option<u32>) -> bool {
    match stable {
        
        _ if version.nightly => false,
        Some(stabilized) if version.minor >= stabilized => false,
        _ => true,
    }
}
fn target_feature_fallback(name: &str, mut has_target_feature: bool) -> bool {
    if let Some(rustflags) = env::var_os("CARGO_ENCODED_RUSTFLAGS") {
        for mut flag in rustflags.to_string_lossy().split('\x1f') {
            flag = strip_prefix(flag, "-C").unwrap_or(flag);
            if let Some(flag) = strip_prefix(flag, "target-feature=") {
                for s in flag.split(',') {
                    
                    
                    match (s.as_bytes().first(), s.as_bytes().get(1..)) {
                        (Some(b'+'), Some(f)) if f == name.as_bytes() => has_target_feature = true,
                        (Some(b'-'), Some(f)) if f == name.as_bytes() => has_target_feature = false,
                        _ => {}
                    }
                }
            }
        }
    }
    if has_target_feature {
        println!("cargo:rustc-cfg=portable_atomic_target_feature=\"{}\"", name);
    }
    has_target_feature
}

fn target_cpu() -> Option<String> {
    let rustflags = env::var_os("CARGO_ENCODED_RUSTFLAGS")?;
    let rustflags = rustflags.to_string_lossy();
    let mut cpu = None;
    for mut flag in rustflags.split('\x1f') {
        flag = strip_prefix(flag, "-C").unwrap_or(flag);
        if let Some(flag) = strip_prefix(flag, "target-cpu=") {
            cpu = Some(flag);
        }
    }
    cpu.map(str::to_owned)
}

fn is_allowed_feature(name: &str) -> bool {
    
    if env::var_os("RUSTC_STAGE").is_some() {
        return false;
    }

    
    let mut allowed = true;
    if let Some(rustflags) = env::var_os("CARGO_ENCODED_RUSTFLAGS") {
        for mut flag in rustflags.to_string_lossy().split('\x1f') {
            flag = strip_prefix(flag, "-Z").unwrap_or(flag);
            if let Some(flag) = strip_prefix(flag, "allow-features=") {
                
                allowed = flag.split(',').any(|allowed| allowed == name);
            }
        }
    }
    allowed
}








fn convert_custom_linux_target(target: &str) -> String {
    let mut parts: Vec<&str> = target.split('-').collect();
    let system = parts.get(2);
    if system == Some(&"linux") {
        parts[1] = "unknown";
    }
    parts.join("-")
}


#[must_use]
fn strip_prefix<'a>(s: &'a str, pat: &str) -> Option<&'a str> {
    if s.starts_with(pat) { Some(&s[pat.len()..]) } else { None }
}

#[must_use]
fn strip_suffix<'a>(s: &'a str, pat: &str) -> Option<&'a str> {
    if s.ends_with(pat) { Some(&s[..s.len() - pat.len()]) } else { None }
}
