#' Estimate precise sleep_wake cycles using CircaCP algorithm
#'
#' It first uses a 24 h cosinor fit (via [sleep_cos()]) and a chosen thresholding rule to
#' label each minute as sleep (`label.sw = 1`) or wake (`label.sw = 0`).
#' Reference: Shanshan Chen, and Xinxin Sun.Validating CircaCP:
#' a generic sleep–wake cycle detection algorithm for unlabelled actigraphy data.
#' Royal Society Open Science 11, no. 5 (2024): 231468.
#'
#' @param clean_df Minute-level `data.frame` with at least `Activity`. Additional
#'   columns are preserved.
#' @param thr Numeric threshold in \[0, 1\] applied to a rescaled cosinor fit.
#' @param dist Character method key (e.g., `"ZAG"`); interpreted by your rule set.
#'
#' @return a data.frame augmenting the input `df` including the following additional variables:
#' \describe{
#'   \item{cosinor}{fitted cosine curve}
#'   \item{label.cos}{circadian cycle estimated by dichotimized cosine curve}
#'   \item{label.sw}{sleep-wake cycle estimated by CircaCP}
#'   \item{Activity_norm}{range-normalized activity levels}
#' }
#'
#' @seealso [screen_wear()], [sleep_cos()],[cp_detect()], [extract_sleep_metrics()]
#'
#' @export


sleep_detection <- function(clean_df, thr=0.2, dist = 'ZAG'){
  M <- abs(clean_df$Activity)
  label.cos <- sleep_cos(clean_df,thr)$label.cos #T in matlab code
  cosinor <- sleep_cos(clean_df,thr = 0.2)$fitted #T in matlab code

  E <- which(diff(label.cos)!=0) #edges detected by cos-sigmoid
  CP <- E
  EWT <- which(diff(label.cos)==1) # wake times determined by cos
  EST <- which(diff(label.cos)==-1) # sleep times determined by cos

  tempmag <- M[1:CP[2]]
  if(length(M[1:CP[1]])>360) {
    gt <- cp_detect((tempmag),dist)
    if(length(gt)>1) {
      test <- abs(gt-CP[1])
      gt <- gt[test==min(test)]
    }
  } else {
    gt <- CP[1]
  }

  ipt <- numeric()
  label_ipt <- numeric()
  ipt[1] <- gt
  if(CP[2] %in% EWT){
    label_ipt[1] <- -1
  } else {
    label_ipt[1] <- 1
  }

  for(i in 2:(length(CP)-1)){
    pre_pt <- ipt[i-1]
    curr_pt <- CP[i]
    next_pt <- CP[i+1]
    tempmag <- M[pre_pt:next_pt]
    pt <- cp_detect(tempmag,dist)
    if(length(pt)>1){
      test <- abs(pt+pre_pt-curr_pt)
      pt <- pt[test==min(test)]
    }
    ipt[i] <- pt[1] + pre_pt
    if(next_pt %in% EWT){
      label_ipt[i] <- -1
    } else {
      label_ipt[i] <- 1
    }

  }#end of for loop

  ct <- numeric()
  tempmag <- M[ipt[length(ipt)]:length(M)]
  if(length(M[CP[length(CP)]:length(M)])>120){
    ct <- cp_detect(tempmag,dist)
    if(length(ct)>1){
      test <- abs(ct+ipt[length(ipt)]-CP[length(CP)])
      ct <- ct[test==min(test)]
    }
    ct <- ct+ipt[length(ipt)]
  } else {
    ct <- CP[length(CP)]
  }
  ipt <- c(ipt,ct)
  Len <- length(ipt)
  if(CP[length(CP)-1] %in% EWT){
    label_ipt[Len] <- -1
  } else {
    label_ipt[Len] <- 1
  }

  AWT <- ipt[label_ipt==1]
  SLT <- ipt[label_ipt==-1]
  L <- CP2label(AWT,SLT,length(M))



  # R <- sleep_para(S,label.cos,L,EWT,EST,AWT,SLT)
  labeled_df = clean_df
  labeled_df$label.cos = label.cos
  labeled_df$label.sw = L
  labeled_df$cosinor = cosinor

  labeled_df$Activity_norm = (labeled_df$Activity - min(labeled_df$Activity))/(max(labeled_df$Activity)-min(labeled_df$Activity))

  return(labeled_df)
} #end of function



